"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertToSerializableGraph = convertToSerializableGraph;
exports.convertToWorkflowGraph = convertToWorkflowGraph;
exports.visitAtomicStep = visitAtomicStep;
exports.visitElasticsearchStep = visitElasticsearchStep;
exports.visitHttpStep = visitHttpStep;
exports.visitKibanaStep = visitKibanaStep;
exports.visitWaitStep = visitWaitStep;
var _dagre = require("@dagrejs/dagre");
var _lodash = require("lodash");
var _create_typed_graph = require("../workflow_graph/create_typed_graph");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const flowControlStepTypes = new Set(['if', 'foreach']);
const disallowedWorkflowLevelOnFailureSteps = new Set(['wait']);

/** Context used during the graph construction to keep track of settings and avoid cycles */

function getStepId(node, context) {
  // TODO: This is a workaround for the fact that some steps do not have an `id` field.
  // We should ensure that all steps have an `id` field in the future - either explicitly set or generated from name.
  const nodeId = node.id || node.name;
  const parts = [];
  if (context.parentKey) {
    parts.push(context.parentKey);
  }
  parts.push(nodeId);
  return parts.join('_');
}
function visitAbstractStep(currentStep, context) {
  var _context$settings, _type, _type2;
  if (currentStep['on-failure']) {
    const stepLevelOnFailureGraph = handleStepLevelOnFailure(currentStep, context);
    if (stepLevelOnFailureGraph) {
      return stepLevelOnFailureGraph;
    }
  }
  if ((_context$settings = context.settings) !== null && _context$settings !== void 0 && _context$settings['on-failure']) {
    const workflowLevelOnFailureGraph = handleWorkflowLevelOnFailure(currentStep, context);
    if (workflowLevelOnFailureGraph) {
      return workflowLevelOnFailureGraph;
    }
  }
  if (currentStep.if) {
    return createIfGraphForIfStepLevel(currentStep, context);
  }
  if (currentStep.type === 'if') {
    return createIfGraph(getStepId(currentStep, context), currentStep, context);
  }
  if (currentStep.type === 'foreach') {
    return createForeachGraph(getStepId(currentStep, context), currentStep, context);
  }
  if (currentStep.foreach) {
    return createForeachGraphForStepWithForeach(currentStep, context);
  }
  if (currentStep.timeout) {
    const step = currentStep;
    return handleTimeout(getStepId(step, context), 'step_level_timeout', step.timeout, visitAbstractStep((0, _lodash.omit)(step, ['timeout']), context), context);
  }
  if (currentStep.type === 'wait') {
    return visitWaitStep(currentStep, context);
  }
  if (currentStep.type === 'http') {
    return visitHttpStep(currentStep, context);
  }
  if ((_type = currentStep.type) !== null && _type !== void 0 && _type.startsWith('elasticsearch.')) {
    return visitElasticsearchStep(currentStep, context);
  }
  if ((_type2 = currentStep.type) !== null && _type2 !== void 0 && _type2.startsWith('kibana.')) {
    return visitKibanaStep(currentStep, context);
  }
  return visitAtomicStep(currentStep, context);
}
function visitWaitStep(currentStep, context) {
  const stepId = getStepId(currentStep, context);
  const graph = (0, _create_typed_graph.createTypedGraph)({
    directed: true
  });
  const waitNode = {
    id: getStepId(currentStep, context),
    type: 'wait',
    stepId,
    stepType: currentStep.type,
    configuration: {
      ...currentStep
    }
  };
  graph.setNode(waitNode.id, waitNode);
  return graph;
}
function visitHttpStep(currentStep, context) {
  const stepId = getStepId(currentStep, context);
  const graph = (0, _create_typed_graph.createTypedGraph)({
    directed: true
  });
  const httpNode = {
    id: getStepId(currentStep, context),
    type: 'http',
    stepId,
    stepType: currentStep.type,
    configuration: {
      ...currentStep
    }
  };
  graph.setNode(httpNode.id, httpNode);
  return graph;
}
function visitElasticsearchStep(currentStep, context) {
  const graph = (0, _create_typed_graph.createTypedGraph)({
    directed: true
  });
  const elasticsearchNode = {
    id: getStepId(currentStep, context),
    stepId: getStepId(currentStep, context),
    stepType: currentStep.type,
    type: currentStep.type,
    // e.g., 'elasticsearch.search.query'
    configuration: {
      ...currentStep
    }
  };
  graph.setNode(elasticsearchNode.id, elasticsearchNode);
  return graph;
}
function visitKibanaStep(currentStep, context) {
  const graph = (0, _create_typed_graph.createTypedGraph)({
    directed: true
  });
  const kibanaNode = {
    id: getStepId(currentStep, context),
    stepId: getStepId(currentStep, context),
    stepType: currentStep.type,
    type: currentStep.type,
    // e.g., 'kibana.cases.create'
    configuration: {
      ...currentStep
    }
  };
  graph.setNode(kibanaNode.id, kibanaNode);
  return graph;
}
function visitAtomicStep(currentStep, context) {
  const stepId = getStepId(currentStep, context);
  const graph = (0, _create_typed_graph.createTypedGraph)({
    directed: true
  });
  const atomicNode = {
    id: getStepId(currentStep, context),
    type: 'atomic',
    stepId,
    stepType: currentStep.type,
    configuration: {
      ...currentStep
    }
  };
  graph.setNode(atomicNode.id, atomicNode);
  return graph;
}
function createIfGraph(stepId, ifStep, context) {
  const graph = (0, _create_typed_graph.createTypedGraph)({
    directed: true
  });
  const enterConditionNodeId = `enterCondition_${stepId}`;
  const exitConditionNodeId = `exitCondition_${stepId}`;
  const ifElseStep = ifStep;
  const trueSteps = ifElseStep.steps || [];
  const falseSteps = ifElseStep.else || [];
  const conditionNode = {
    id: enterConditionNodeId,
    exitNodeId: exitConditionNodeId,
    type: 'enter-if',
    stepId,
    stepType: ifStep.type,
    configuration: {
      ...(0, _lodash.omit)(ifElseStep, ['steps', 'else']) // No need to include them as they will be represented in the graph
    }
  };
  context.stack.push(conditionNode);
  const exitConditionNode = {
    type: 'exit-if',
    id: exitConditionNodeId,
    stepId,
    stepType: ifStep.type,
    startNodeId: enterConditionNodeId
  };
  const enterThenBranchNode = {
    id: `enterThen_${stepId}`,
    type: 'enter-then-branch',
    condition: ifElseStep.condition,
    stepId,
    stepType: ifStep.type
  };
  graph.setNode(enterThenBranchNode.id, enterThenBranchNode);
  graph.setEdge(enterConditionNodeId, enterThenBranchNode.id);
  const exitThenBranchNode = {
    id: `exitThen_${stepId}`,
    type: 'exit-then-branch',
    stepType: ifStep.type,
    startNodeId: enterThenBranchNode.id,
    stepId
  };
  const thenGraph = createStepsSequence(trueSteps, context);
  insertGraphBetweenNodes(graph, thenGraph, enterThenBranchNode.id, exitThenBranchNode.id);
  graph.setNode(exitThenBranchNode.id, exitThenBranchNode);
  graph.setEdge(exitThenBranchNode.id, exitConditionNode.id);
  if ((falseSteps === null || falseSteps === void 0 ? void 0 : falseSteps.length) > 0) {
    const enterElseBranchNode = {
      id: `enterElse_${stepId}`,
      type: 'enter-else-branch',
      stepId,
      stepType: ifStep.type
    };
    graph.setNode(enterElseBranchNode.id, enterElseBranchNode);
    graph.setEdge(enterConditionNodeId, enterElseBranchNode.id);
    const exitElseBranchNode = {
      id: `exitElse_${stepId}`,
      type: 'exit-else-branch',
      startNodeId: enterElseBranchNode.id,
      stepId,
      stepType: ifStep.type
    };
    const elseGraph = createStepsSequence(falseSteps, context);
    insertGraphBetweenNodes(graph, elseGraph, enterElseBranchNode.id, exitElseBranchNode.id);
    graph.setNode(exitElseBranchNode.id, exitElseBranchNode);
    graph.setEdge(exitElseBranchNode.id, exitConditionNode.id);
  }
  graph.setNode(exitConditionNode.id, exitConditionNode);
  graph.setNode(enterConditionNodeId, conditionNode);
  context.stack.pop();
  return graph;
}
function createIfGraphForIfStepLevel(stepWithIfCondition, context) {
  const stepId = getStepId(stepWithIfCondition, context);
  const generatedStepId = `if_${stepId}`;
  const ifStep = {
    name: generatedStepId,
    type: 'if',
    condition: stepWithIfCondition.if,
    steps: [(0, _lodash.omit)(stepWithIfCondition, ['if'])]
  };
  return createIfGraph(generatedStepId, ifStep, context);
}
function visitOnFailure(currentStep, onFailureConfiguration, context) {
  var _onFailureConfigurati;
  const stepId = getStepId(currentStep, context);
  const onFailureGraphNode = {
    id: `onFailure_${stepId}`,
    type: 'on-failure',
    stepId,
    stepType: 'on-failure'
  };
  context.stack.push(onFailureGraphNode);
  let graph = createStepsSequence([{
    ...currentStep,
    ['on-failure']: undefined // Remove 'on-failure' to avoid infinite recursion
  }], context);
  if (onFailureConfiguration !== null && onFailureConfiguration !== void 0 && onFailureConfiguration.retry) {
    graph = createRetry(stepId, graph, onFailureConfiguration.retry);
  }
  if ((_onFailureConfigurati = onFailureConfiguration.fallback) !== null && _onFailureConfigurati !== void 0 && _onFailureConfigurati.length) {
    graph = createFallback(stepId, graph, onFailureConfiguration.fallback, context);
  }
  if (onFailureConfiguration.continue) {
    graph = createContinue(stepId, graph);
  }
  context.stack.pop();
  return graph;
}
function handleTimeout(stepId, stepType, timeout, innerGraph, context) {
  const enterTimeoutZone = {
    id: `enterTimeoutZone_${stepId}`,
    type: 'enter-timeout-zone',
    stepId,
    stepType,
    timeout
  };
  const exitTimeoutZone = {
    id: `exitTimeoutZone_${stepId}`,
    type: 'exit-timeout-zone',
    stepId,
    stepType
  };
  const graph = new _dagre.graphlib.Graph({
    directed: true
  });
  graph.setNode(enterTimeoutZone.id, enterTimeoutZone);
  graph.setNode(exitTimeoutZone.id, exitTimeoutZone);
  context.stack.push(enterTimeoutZone);
  insertGraphBetweenNodes(graph, innerGraph, enterTimeoutZone.id, exitTimeoutZone.id);
  context.stack.pop();
  return graph;
}
function handleStepLevelOnFailure(step, context) {
  const stackEntry = {
    id: `stepLevelOnFailure_${getStepId(step, context)}`,
    type: 'step-level-on-failure',
    stepId: getStepId(step, context),
    stepType: step.type
  };
  if (context.stack.some(node => node.id === stackEntry.id)) {
    return null;
  }
  context.stack.push(stackEntry);
  const result = visitOnFailure(step, step['on-failure'], context);
  context.stack.pop();
  return result;
}
function handleWorkflowLevelOnFailure(step, context) {
  if (flowControlStepTypes.has(step.type) || disallowedWorkflowLevelOnFailureSteps.has(step.type)) {
    return null;
  }
  const stackEntry = {
    id: `workflowLevelOnFailure_${getStepId(step, {
      ...context,
      parentKey: ''
    })}`,
    type: 'workflow-level-on-failure',
    stepId: getStepId(step, {
      ...context,
      parentKey: ''
    }),
    stepType: step.name
  };
  if (context.stack.some(node => node.id === stackEntry.id) ||
  // Avoid recursion
  context.stack.some(node => node.id === `stepLevelOnFailure_${getStepId(step, context)}`) ||
  // Avoid workflow-level on-failure if already in step-level on-failure
  context.stack.some(node => node.type === 'enter-fallback-path') // Avoid workflow-level on-failure for steps inside fallback path
  ) {
    return null;
  }
  context.stack.push(stackEntry);
  const result = visitOnFailure(step, context.settings['on-failure'], context);
  context.stack.pop();
  return result;
}
function createContinue(stepId, innerGraph) {
  const graph = (0, _create_typed_graph.createTypedGraph)({
    directed: true
  });
  const enterContinueNodeId = `enterContinue_${stepId}`;
  const exitNodeId = `exitContinue_${stepId}`;
  const enterContinueNode = {
    id: enterContinueNodeId,
    type: 'enter-continue',
    stepId,
    stepType: 'continue',
    exitNodeId
  };
  const exitContinueNode = {
    type: 'exit-continue',
    stepId,
    stepType: 'continue',
    id: exitNodeId
  };
  graph.setNode(enterContinueNode.id, enterContinueNode);
  graph.setNode(exitContinueNode.id, exitContinueNode);
  insertGraphBetweenNodes(graph, innerGraph, enterContinueNode.id, exitContinueNode.id);
  return graph;
}
function createRetry(stepId, innerGraph, retry) {
  const graph = (0, _create_typed_graph.createTypedGraph)({
    directed: true
  });
  const enterRetryNodeId = `enterRetry_${stepId}`;
  const exitNodeId = `exitRetry_${stepId}`;
  const enterRetryNode = {
    id: enterRetryNodeId,
    type: 'enter-retry',
    stepId,
    stepType: 'retry',
    exitNodeId,
    configuration: retry
  };
  const exitRetryNode = {
    type: 'exit-retry',
    id: exitNodeId,
    stepId,
    stepType: 'retry',
    startNodeId: enterRetryNodeId
  };
  graph.setNode(enterRetryNode.id, enterRetryNode);
  graph.setNode(exitRetryNode.id, exitRetryNode);
  insertGraphBetweenNodes(graph, innerGraph, enterRetryNode.id, exitRetryNode.id);
  return graph;
}
function createNormalPath(stepId, normalPathGraph) {
  const graph = (0, _create_typed_graph.createTypedGraph)({
    directed: true
  });
  const enterNormalPathNodeId = `enterNormalPath_${stepId}`;
  const exitNormalPathNodeId = `exitNormalPath_${stepId}`;
  const enterNormalPathNode = {
    id: enterNormalPathNodeId,
    type: 'enter-normal-path',
    stepId,
    stepType: 'fallback',
    enterZoneNodeId: `enterTryBlock_${stepId}`,
    enterFailurePathNodeId: `enterFallbackPath_${stepId}`
  };
  const exitNormalPathNode = {
    id: exitNormalPathNodeId,
    stepId,
    stepType: 'fallback',
    type: 'exit-normal-path',
    enterNodeId: enterNormalPathNodeId,
    exitOnFailureZoneNodeId: `exitTryBlock_${stepId}`
  };
  graph.setNode(enterNormalPathNode.id, enterNormalPathNode);
  graph.setNode(exitNormalPathNode.id, exitNormalPathNode);
  insertGraphBetweenNodes(graph, normalPathGraph, enterNormalPathNode.id, exitNormalPathNode.id);
  return graph;
}
function createFallbackPath(stepId, fallbackSteps, context) {
  const workflowLevelOnFailure = context.stack.find(node => node.type === 'workflow-level-on-failure');
  const graph = (0, _create_typed_graph.createTypedGraph)({
    directed: true
  });
  const enterFallbackPathNodeId = `enterFallbackPath_${stepId}`;
  const exitFallbackPathNodeId = `exitFallbackPath_${stepId}`;
  const enterFallbackPathNode = {
    id: enterFallbackPathNodeId,
    stepId,
    stepType: 'fallback',
    type: 'enter-fallback-path',
    enterZoneNodeId: enterFallbackPathNodeId
  };
  context.stack.push(enterFallbackPathNode);
  const exitFallbackPathNode = {
    id: exitFallbackPathNodeId,
    stepId,
    stepType: 'fallback',
    type: 'exit-fallback-path',
    enterNodeId: enterFallbackPathNodeId,
    exitOnFailureZoneNodeId: `exitTryBlock_${stepId}`
  };
  graph.setNode(enterFallbackPathNode.id, enterFallbackPathNode);
  graph.setNode(exitFallbackPathNode.id, exitFallbackPathNode);
  const fallbackPathGraph = createStepsSequence(fallbackSteps, {
    ...context,
    parentKey: workflowLevelOnFailure ? [workflowLevelOnFailure.type, stepId].join('_') : ''
  });
  insertGraphBetweenNodes(graph, fallbackPathGraph, enterFallbackPathNode.id, exitFallbackPathNode.id);
  context.stack.pop();
  return graph;
}
function createFallback(stepId, normalPathGraph, fallbackPathSteps, context) {
  const graph = (0, _create_typed_graph.createTypedGraph)({
    directed: true
  });
  const enterTryBlockNodeId = `enterTryBlock_${stepId}`;
  const exitTryBlockNodeId = `exitTryBlock_${stepId}`;
  const enterNormalPathNodeId = `enterNormalPath_${stepId}`;
  const enterTryBlockNode = {
    id: enterTryBlockNodeId,
    exitNodeId: exitTryBlockNodeId,
    stepId,
    stepType: 'fallback',
    type: 'enter-try-block',
    enterNormalPathNodeId
  };
  graph.setNode(enterTryBlockNodeId, enterTryBlockNode);
  const exitTryBlockNode = {
    id: exitTryBlockNodeId,
    type: 'exit-try-block',
    stepId,
    stepType: 'fallback',
    enterNodeId: enterTryBlockNodeId
  };
  graph.setNode(exitTryBlockNodeId, exitTryBlockNode);
  const normalPathGraphWithNodes = createNormalPath(stepId, normalPathGraph);
  insertGraphBetweenNodes(graph, normalPathGraphWithNodes, enterTryBlockNodeId, exitTryBlockNodeId);
  const fallbackPathGraph = createFallbackPath(stepId, fallbackPathSteps, context);
  insertGraphBetweenNodes(graph, fallbackPathGraph, enterTryBlockNodeId, exitTryBlockNodeId);
  return graph;
}
function createStepsSequence(steps, context) {
  const graph = (0, _create_typed_graph.createTypedGraph)({
    directed: true
  });
  let previousGraph = null;
  for (let i = 0; i < steps.length; i++) {
    const currentGraph = visitAbstractStep(steps[i], context);
    currentGraph.nodes().forEach(nodeId => {
      graph.setNode(nodeId, currentGraph.node(nodeId));
    });
    currentGraph.edges().forEach(edgeObj => {
      graph.setEdge(edgeObj.v, edgeObj.w);
    });
    if (previousGraph) {
      const previousEndNodes = previousGraph.nodes().filter(nodeId => {
        var _outEdges;
        return ((_outEdges = previousGraph.outEdges(nodeId)) === null || _outEdges === void 0 ? void 0 : _outEdges.length) === 0;
      });
      const currentStartNodes = currentGraph.nodes().filter(nodeId => {
        var _currentGraph$inEdges;
        return ((_currentGraph$inEdges = currentGraph.inEdges(nodeId)) === null || _currentGraph$inEdges === void 0 ? void 0 : _currentGraph$inEdges.length) === 0;
      });
      previousEndNodes.forEach(endNode => {
        currentStartNodes.forEach(startNode => {
          graph.setEdge(endNode, startNode);
        });
      });
    }
    previousGraph = currentGraph;
  }
  return graph;
}
function insertGraphBetweenNodes(graph, subGraph, startNodeId, endNodeId) {
  // Find all start nodes (no incoming edges) and end nodes (no outgoing edges)
  const startNodes = subGraph.nodes().filter(nodeId => {
    var _subGraph$inEdges;
    return ((_subGraph$inEdges = subGraph.inEdges(nodeId)) === null || _subGraph$inEdges === void 0 ? void 0 : _subGraph$inEdges.length) === 0;
  });
  const endNodes = subGraph.nodes().filter(nodeId => {
    var _subGraph$outEdges;
    return ((_subGraph$outEdges = subGraph.outEdges(nodeId)) === null || _subGraph$outEdges === void 0 ? void 0 : _subGraph$outEdges.length) === 0;
  });

  // Connect all start nodes to the main start node
  startNodes.forEach(startNode => {
    graph.setEdge(startNodeId, startNode);
  });

  // Connect all end nodes to the main end node
  endNodes.forEach(endNode => {
    graph.setEdge(endNode, endNodeId);
  });

  // Copy all nodes from subGraph to the main graph
  subGraph.nodes().forEach(nodeId => {
    graph.setNode(nodeId, subGraph.node(nodeId));
  });

  // Copy all edges from subGraph to the main graph
  subGraph.edges().forEach(edgeObj => {
    graph.setEdge(edgeObj.v, edgeObj.w);
  });
}
function createForeachGraph(stepId, foreachStep, context) {
  const graph = (0, _create_typed_graph.createTypedGraph)({
    directed: true
  });
  const enterForeachNodeId = `enterForeach_${stepId}`;
  const exitNodeId = `exitForeach_${stepId}`;
  const enterForeachNode = {
    id: enterForeachNodeId,
    type: 'enter-foreach',
    stepId,
    stepType: foreachStep.type,
    exitNodeId,
    configuration: {
      ...(0, _lodash.omit)(foreachStep, ['steps']) // No need to include them as they will be represented in the graph
    }
  };
  context.stack.push(enterForeachNode);
  graph.setNode(enterForeachNodeId, enterForeachNode);
  const exitForeachNode = {
    type: 'exit-foreach',
    id: exitNodeId,
    stepType: foreachStep.type,
    stepId,
    startNodeId: enterForeachNodeId
  };
  graph.setNode(exitNodeId, exitForeachNode);
  const innerGraph = createStepsSequence(foreachStep.steps || [], context);
  insertGraphBetweenNodes(graph, innerGraph, enterForeachNodeId, exitNodeId);
  context.stack.pop();
  return graph;
}
function createForeachGraphForStepWithForeach(stepWithForeach, context) {
  const stepId = getStepId(stepWithForeach, context);
  const generatedStepId = `foreach_${stepId}`;
  const foreachStep = {
    name: generatedStepId,
    type: 'foreach',
    foreach: stepWithForeach.foreach,
    steps: [(0, _lodash.omit)(stepWithForeach, ['foreach'])]
  };
  return createForeachGraph(generatedStepId, foreachStep, context);
}
function convertToWorkflowGraph(workflowSchema) {
  var _workflowSchema$setti;
  const context = {
    settings: workflowSchema.settings,
    stack: [],
    parentKey: ''
  };
  let finalGraph = createStepsSequence(workflowSchema.steps, context);
  if ((_workflowSchema$setti = workflowSchema.settings) !== null && _workflowSchema$setti !== void 0 && _workflowSchema$setti.timeout) {
    finalGraph = handleTimeout('workflow_level_timeout', 'workflow_level_timeout', workflowSchema.settings.timeout, finalGraph, context);
  }
  return finalGraph;
}
function convertToSerializableGraph(graph) {
  return _dagre.graphlib.json.write(graph); // GraphLib does not provide type information, so we use `any`
}