"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getForEachStepSchema = exports.WorkflowSettingsSchema = exports.WorkflowSchema = exports.WorkflowRetrySchema = exports.WorkflowOnFailureSchema = exports.WorkflowInputTypeEnum = exports.WorkflowInputSchema = exports.WorkflowExecutionContextSchema = exports.WorkflowDataContextSchema = exports.WorkflowContextSchema = exports.WorkflowConstsSchema = exports.WaitStepSchema = exports.TriggerTypes = exports.TriggerSchema = exports.TimeoutPropSchema = exports.StepWithOnFailureSchema = exports.StepDataSchema = exports.StepContextSchema = exports.ScheduledTriggerSchema = exports.RetryPolicySchema = exports.ParallelStepSchema = exports.MergeStepSchema = exports.ManualTriggerSchema = exports.KibanaStepSchema = exports.IfStepSchema = exports.HttpStepSchema = exports.ForEachStepSchema = exports.ForEachContextSchema = exports.EventSchema = exports.ElasticsearchStepSchema = exports.DynamicWorkflowContextSchema = exports.DynamicStepContextSchema = exports.DurationSchema = exports.BuiltInStepTypes = exports.BaseStepSchema = exports.BaseConnectorStepSchema = exports.AlertRuleTriggerSchema = void 0;
exports.getHttpStepSchema = getHttpStepSchema;
exports.getMergeStepSchema = exports.getIfStepSchema = void 0;
exports.getOnFailureStepSchema = getOnFailureStepSchema;
exports.getParallelStepSchema = void 0;
exports.getWorkflowSettingsSchema = getWorkflowSettingsSchema;
var _zod = require("@kbn/zod");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const DurationSchema = exports.DurationSchema = _zod.z.string().regex(/^\d+(ms|[smhdw])$/, 'Invalid duration format');

/* -- Settings -- */
const RetryPolicySchema = exports.RetryPolicySchema = _zod.z.object({
  'max-attempts': _zod.z.number().int().min(1).optional(),
  'timeout-seconds': _zod.z.number().int().min(1).optional()
});
const WorkflowRetrySchema = exports.WorkflowRetrySchema = _zod.z.object({
  'max-attempts': _zod.z.number().min(1),
  delay: _zod.z.string().regex(/^\d+(ms|[smhdw])$/, 'Invalid duration format').optional() // e.g., '5s', '1m', '2h' (default: no delay)
});
// Base step schema, with recursive steps property
const BaseStepSchema = exports.BaseStepSchema = _zod.z.object({
  name: _zod.z.string().min(1),
  type: _zod.z.string()
});
const WorkflowOnFailureSchema = exports.WorkflowOnFailureSchema = _zod.z.object({
  retry: WorkflowRetrySchema.optional(),
  fallback: _zod.z.array(BaseStepSchema).min(1).optional(),
  continue: _zod.z.boolean().optional()
});
function getOnFailureStepSchema(stepSchema, loose = false) {
  const schema = WorkflowOnFailureSchema.extend({
    fallback: _zod.z.array(stepSchema).optional()
  });
  if (loose) {
    // make all fields optional, but require type to be present for discriminated union
    return schema.partial();
  }
  return schema;
}
const WorkflowSettingsSchema = exports.WorkflowSettingsSchema = _zod.z.object({
  'on-failure': WorkflowOnFailureSchema.optional(),
  timezone: _zod.z.string().optional(),
  // Should follow IANA TZ format
  timeout: DurationSchema.optional() // e.g., '5s', '1m', '2h'
});
function getWorkflowSettingsSchema(stepSchema, loose = false) {
  const schema = WorkflowSettingsSchema.extend({
    'on-failure': getOnFailureStepSchema(stepSchema, loose).optional()
  });
  if (loose) {
    // make all fields optional, but require type to be present for discriminated union
    return schema.partial();
  }
  return schema;
}

/* --- Triggers --- */
const AlertRuleTriggerSchema = exports.AlertRuleTriggerSchema = _zod.z.object({
  type: _zod.z.literal('alert'),
  enabled: _zod.z.boolean().optional().default(true),
  with: _zod.z.union([_zod.z.object({
    rule_id: _zod.z.string().min(1)
  }), _zod.z.object({
    rule_name: _zod.z.string().min(1)
  })]).optional()
});
const ScheduledTriggerSchema = exports.ScheduledTriggerSchema = _zod.z.object({
  type: _zod.z.literal('scheduled'),
  enabled: _zod.z.boolean().optional().default(true),
  with: _zod.z.union([_zod.z.object({
    every: _zod.z.string().min(1),
    unit: _zod.z.enum(['second', 'minute', 'hour', 'day', 'week', 'month', 'year'])
  }), _zod.z.object({
    cron: _zod.z.string().min(1)
  })])
});
const ManualTriggerSchema = exports.ManualTriggerSchema = _zod.z.object({
  type: _zod.z.literal('manual'),
  enabled: _zod.z.boolean().optional().default(true)
});
const TriggerSchema = exports.TriggerSchema = _zod.z.discriminatedUnion('type', [AlertRuleTriggerSchema, ScheduledTriggerSchema, ManualTriggerSchema]);
const TriggerTypes = exports.TriggerTypes = [AlertRuleTriggerSchema.shape.type._def.value, ScheduledTriggerSchema.shape.type._def.value, ManualTriggerSchema.shape.type._def.value];
/* --- Steps --- */
const TimeoutPropSchema = exports.TimeoutPropSchema = _zod.z.object({
  timeout: DurationSchema.optional()
});
const StepWithForEachSchema = _zod.z.object({
  foreach: _zod.z.string().optional()
});
const StepWithIfConditionSchema = _zod.z.object({
  if: _zod.z.string().optional()
});
const StepWithOnFailureSchema = exports.StepWithOnFailureSchema = _zod.z.object({
  'on-failure': WorkflowOnFailureSchema.optional()
});
const BaseConnectorStepSchema = exports.BaseConnectorStepSchema = BaseStepSchema.extend({
  type: _zod.z.string().min(1),
  'connector-id': _zod.z.string().optional(),
  // http.request for example, doesn't need connectorId
  with: _zod.z.record(_zod.z.string(), _zod.z.any()).optional()
}).merge(StepWithIfConditionSchema).merge(StepWithForEachSchema).merge(TimeoutPropSchema).merge(StepWithOnFailureSchema);
const WaitStepSchema = exports.WaitStepSchema = BaseStepSchema.extend({
  type: _zod.z.literal('wait'),
  with: _zod.z.object({
    duration: DurationSchema // e.g., '5s', '1m', '2h'
  })
});
const HttpStepSchema = exports.HttpStepSchema = BaseStepSchema.extend({
  type: _zod.z.literal('http'),
  with: _zod.z.object({
    url: _zod.z.string().min(1),
    method: _zod.z.enum(['GET', 'POST', 'PUT', 'DELETE', 'PATCH']).optional().default('GET'),
    headers: _zod.z.record(_zod.z.string(), _zod.z.union([_zod.z.string(), _zod.z.number(), _zod.z.boolean()])).optional().default({}),
    body: _zod.z.any().optional(),
    timeout: _zod.z.string().optional().default('30s')
  })
}).merge(StepWithIfConditionSchema).merge(StepWithForEachSchema).merge(TimeoutPropSchema).merge(StepWithOnFailureSchema);
// Generic Elasticsearch step schema for backend validation
const ElasticsearchStepSchema = exports.ElasticsearchStepSchema = BaseStepSchema.extend({
  type: _zod.z.string().refine(val => val.startsWith('elasticsearch.'), {
    message: 'Elasticsearch step type must start with "elasticsearch."'
  }),
  with: _zod.z.union([
  // Raw API format - like Dev Console
  _zod.z.object({
    request: _zod.z.object({
      method: _zod.z.enum(['GET', 'POST', 'PUT', 'DELETE', 'PATCH', 'HEAD']).optional().default('GET'),
      path: _zod.z.string().min(1),
      body: _zod.z.any().optional()
    })
  }),
  // Sugar syntax for common operations
  _zod.z.object({
    index: _zod.z.string().optional(),
    id: _zod.z.string().optional(),
    query: _zod.z.record(_zod.z.string(), _zod.z.any()).optional(),
    body: _zod.z.record(_zod.z.string(), _zod.z.any()).optional(),
    size: _zod.z.number().optional(),
    from: _zod.z.number().optional(),
    sort: _zod.z.array(_zod.z.any()).optional(),
    _source: _zod.z.union([_zod.z.boolean(), _zod.z.array(_zod.z.string()), _zod.z.string()]).optional(),
    aggs: _zod.z.record(_zod.z.string(), _zod.z.any()).optional(),
    aggregations: _zod.z.record(_zod.z.string(), _zod.z.any()).optional()
  }).and(_zod.z.record(_zod.z.string(), _zod.z.any())) // Allow additional properties for flexibility
  ])
});
// Generic Kibana step schema for backend validation
const KibanaStepSchema = exports.KibanaStepSchema = BaseStepSchema.extend({
  type: _zod.z.string().refine(val => val.startsWith('kibana.'), {
    message: 'Kibana step type must start with "kibana."'
  }),
  with: _zod.z.union([
  // Raw API format - direct HTTP API calls
  _zod.z.object({
    request: _zod.z.object({
      method: _zod.z.enum(['GET', 'POST', 'PUT', 'DELETE', 'PATCH', 'HEAD']).optional().default('GET'),
      path: _zod.z.string().min(1),
      body: _zod.z.any().optional(),
      headers: _zod.z.record(_zod.z.string(), _zod.z.string()).optional()
    })
  }),
  // Sugar syntax for common Kibana operations
  _zod.z.object({
    // Cases API
    title: _zod.z.string().optional(),
    description: _zod.z.string().optional(),
    tags: _zod.z.array(_zod.z.string()).optional(),
    severity: _zod.z.enum(['low', 'medium', 'high', 'critical']).optional(),
    assignees: _zod.z.array(_zod.z.string()).optional(),
    owner: _zod.z.string().optional(),
    connector: _zod.z.record(_zod.z.string(), _zod.z.any()).optional(),
    settings: _zod.z.record(_zod.z.string(), _zod.z.any()).optional(),
    // Generic parameters
    id: _zod.z.string().optional(),
    case_id: _zod.z.string().optional(),
    space_id: _zod.z.string().optional(),
    page: _zod.z.number().optional(),
    perPage: _zod.z.number().optional(),
    status: _zod.z.string().optional()
  }).and(_zod.z.record(_zod.z.string(), _zod.z.any())) // Allow additional properties for flexibility
  ])
});
function getHttpStepSchema(stepSchema, loose = false) {
  const schema = HttpStepSchema.extend({
    'on-failure': getOnFailureStepSchema(stepSchema, loose).optional()
  });
  if (loose) {
    // make all fields optional, but require type to be present for discriminated union
    return schema.partial().required({
      type: true
    });
  }
  return schema;
}
const ForEachStepSchema = exports.ForEachStepSchema = BaseStepSchema.extend({
  type: _zod.z.literal('foreach'),
  foreach: _zod.z.string(),
  steps: _zod.z.array(BaseStepSchema).min(1)
}).merge(StepWithIfConditionSchema);
const getForEachStepSchema = (stepSchema, loose = false) => {
  const schema = ForEachStepSchema.extend({
    steps: _zod.z.array(stepSchema).min(1),
    'on-failure': getOnFailureStepSchema(stepSchema, loose).optional()
  });
  if (loose) {
    // make all fields optional, but require type to be present for discriminated union
    return schema.partial().required({
      type: true
    });
  }
  return schema;
};
exports.getForEachStepSchema = getForEachStepSchema;
const IfStepSchema = exports.IfStepSchema = BaseStepSchema.extend({
  type: _zod.z.literal('if'),
  condition: _zod.z.string(),
  steps: _zod.z.array(BaseStepSchema).min(1),
  else: _zod.z.array(BaseStepSchema).optional()
});
const getIfStepSchema = (stepSchema, loose = false) => {
  const schema = IfStepSchema.extend({
    steps: _zod.z.array(stepSchema).min(1),
    else: _zod.z.array(stepSchema).optional()
  });
  if (loose) {
    // make all fields optional, but require type to be present for discriminated union
    return schema.partial().required({
      type: true
    });
  }
  return schema;
};
exports.getIfStepSchema = getIfStepSchema;
const ParallelStepSchema = exports.ParallelStepSchema = BaseStepSchema.extend({
  type: _zod.z.literal('parallel'),
  branches: _zod.z.array(_zod.z.object({
    name: _zod.z.string(),
    steps: _zod.z.array(BaseStepSchema)
  }))
});
const getParallelStepSchema = (stepSchema, loose = false) => {
  const schema = ParallelStepSchema.extend({
    branches: _zod.z.array(_zod.z.object({
      name: _zod.z.string(),
      steps: _zod.z.array(stepSchema)
    }))
  });
  if (loose) {
    // make all fields optional, but require type to be present for discriminated union
    return schema.partial().required({
      type: true
    });
  }
  return schema;
};
exports.getParallelStepSchema = getParallelStepSchema;
const MergeStepSchema = exports.MergeStepSchema = BaseStepSchema.extend({
  type: _zod.z.literal('merge'),
  sources: _zod.z.array(_zod.z.string()),
  // references to branches or steps to merge
  steps: _zod.z.array(BaseStepSchema) // steps to run after merge
});
const getMergeStepSchema = (stepSchema, loose = false) => {
  const schema = MergeStepSchema.extend({
    steps: _zod.z.array(stepSchema) // steps to run after merge
  });
  if (loose) {
    // make all fields optional, but require type to be present for discriminated union
    return schema.partial().required({
      type: true
    });
  }
  return schema;
};

/* --- Inputs --- */
exports.getMergeStepSchema = getMergeStepSchema;
const WorkflowInputTypeEnum = exports.WorkflowInputTypeEnum = _zod.z.enum(['string', 'number', 'boolean', 'choice']);
const WorkflowInputBaseSchema = _zod.z.object({
  name: _zod.z.string(),
  description: _zod.z.string().optional(),
  default: _zod.z.any().optional(),
  required: _zod.z.boolean().optional()
});
const WorkflowInputStringSchema = WorkflowInputBaseSchema.extend({
  type: _zod.z.literal('string'),
  default: _zod.z.string().optional()
});
const WorkflowInputNumberSchema = WorkflowInputBaseSchema.extend({
  type: _zod.z.literal('number'),
  default: _zod.z.number().optional()
});
const WorkflowInputBooleanSchema = WorkflowInputBaseSchema.extend({
  type: _zod.z.literal('boolean'),
  default: _zod.z.boolean().optional()
});
const WorkflowInputChoiceSchema = WorkflowInputBaseSchema.extend({
  type: _zod.z.literal('choice'),
  default: _zod.z.string().optional(),
  options: _zod.z.array(_zod.z.string())
});
const WorkflowInputSchema = exports.WorkflowInputSchema = _zod.z.discriminatedUnion('type', [WorkflowInputStringSchema, WorkflowInputNumberSchema, WorkflowInputBooleanSchema, WorkflowInputChoiceSchema]);

/* --- Consts --- */
const WorkflowConstsSchema = exports.WorkflowConstsSchema = _zod.z.record(_zod.z.string(), _zod.z.union([_zod.z.string(), _zod.z.number(), _zod.z.boolean(), _zod.z.record(_zod.z.string(), _zod.z.any()), _zod.z.object({}), _zod.z.array(_zod.z.any())]));
const StepSchema = _zod.z.lazy(() => _zod.z.discriminatedUnion('type', [ForEachStepSchema, IfStepSchema, WaitStepSchema, HttpStepSchema, ElasticsearchStepSchema, KibanaStepSchema, ParallelStepSchema, MergeStepSchema, BaseConnectorStepSchema]));
const BuiltInStepTypes = exports.BuiltInStepTypes = [ForEachStepSchema.shape.type._def.value, IfStepSchema.shape.type._def.value, ParallelStepSchema.shape.type._def.value, MergeStepSchema.shape.type._def.value, WaitStepSchema.shape.type._def.value, HttpStepSchema.shape.type._def.value];
/* --- Workflow --- */
const WorkflowSchema = exports.WorkflowSchema = _zod.z.object({
  version: _zod.z.literal('1').default('1').describe('The version of the workflow schema'),
  name: _zod.z.string().min(1),
  description: _zod.z.string().optional(),
  settings: WorkflowSettingsSchema.optional(),
  enabled: _zod.z.boolean().default(true),
  tags: _zod.z.array(_zod.z.string()).optional(),
  triggers: _zod.z.array(TriggerSchema).min(1),
  inputs: _zod.z.array(WorkflowInputSchema).optional(),
  consts: WorkflowConstsSchema.optional(),
  steps: _zod.z.array(StepSchema).min(1)
});
const WorkflowExecutionContextSchema = exports.WorkflowExecutionContextSchema = _zod.z.object({
  id: _zod.z.string(),
  isTestRun: _zod.z.boolean(),
  startedAt: _zod.z.date()
});
const WorkflowDataContextSchema = exports.WorkflowDataContextSchema = _zod.z.object({
  id: _zod.z.string(),
  name: _zod.z.string(),
  enabled: _zod.z.boolean(),
  spaceId: _zod.z.string()
});
// TODO: import AlertSchema from from '@kbn/alerts-as-data-utils' once it exported, now only type is exported
const AlertSchema = _zod.z.object({
  _id: _zod.z.string(),
  _index: _zod.z.string(),
  kibana: _zod.z.object({
    alert: _zod.z.any()
  }),
  '@timestamp': _zod.z.string()
});
const SummarizedAlertsChunkSchema = _zod.z.object({
  count: _zod.z.number(),
  data: _zod.z.array(_zod.z.union([AlertSchema, _zod.z.any()]))
});
const RuleSchema = _zod.z.object({
  id: _zod.z.string(),
  name: _zod.z.string(),
  tags: _zod.z.array(_zod.z.string()),
  consumer: _zod.z.string(),
  producer: _zod.z.string(),
  ruleTypeId: _zod.z.string()
});
const EventSchema = exports.EventSchema = _zod.z.object({
  alerts: _zod.z.object({
    new: SummarizedAlertsChunkSchema,
    ongoing: SummarizedAlertsChunkSchema,
    recovered: SummarizedAlertsChunkSchema,
    all: SummarizedAlertsChunkSchema
  }),
  rule: RuleSchema,
  spaceId: _zod.z.string(),
  params: _zod.z.any()
});
const WorkflowContextSchema = exports.WorkflowContextSchema = _zod.z.object({
  event: EventSchema.optional(),
  execution: WorkflowExecutionContextSchema,
  workflow: WorkflowDataContextSchema,
  inputs: _zod.z.record(_zod.z.union([_zod.z.string(), _zod.z.number(), _zod.z.boolean()])).optional(),
  consts: _zod.z.record(_zod.z.string(), _zod.z.any()).optional(),
  now: _zod.z.date().optional()
});
const DynamicWorkflowContextSchema = exports.DynamicWorkflowContextSchema = WorkflowContextSchema.extend({
  // overriding record with object to avoid type mismatch when
  // extending with actual inputs and consts of different types
  inputs: _zod.z.object({}),
  consts: _zod.z.object({})
});
const StepDataSchema = exports.StepDataSchema = _zod.z.object({
  output: _zod.z.any().optional(),
  error: _zod.z.any().optional()
});
const ForEachContextItemSchema = _zod.z.unknown();
const ForEachContextSchema = exports.ForEachContextSchema = _zod.z.object({
  items: _zod.z.array(ForEachContextItemSchema),
  index: _zod.z.number().int(),
  item: ForEachContextItemSchema,
  total: _zod.z.number().int()
});
const StepContextSchema = exports.StepContextSchema = WorkflowContextSchema.extend({
  steps: _zod.z.record(_zod.z.string(), StepDataSchema),
  foreach: ForEachContextSchema.optional()
});
const DynamicStepContextSchema = exports.DynamicStepContextSchema = DynamicWorkflowContextSchema.extend({
  // overriding record with object to avoid type mismatch when
  // extending with actual step ids and different output types
  steps: _zod.z.object({})
});