"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const js_sdk_common_1 = require("@launchdarkly/js-sdk-common");
/**
 * @internal
 */
class Requestor {
    constructor(config, _requests, baseHeaders, _path = '/sdk/latest-all', _logger) {
        this._requests = _requests;
        this._path = _path;
        this._logger = _logger;
        this._eTagCache = {};
        this._headers = Object.assign({}, baseHeaders);
        this._serviceEndpoints = config.serviceEndpoints;
    }
    /**
     * Perform a request and utilize the ETag cache. The ETags are cached in the
     * requestor instance.
     */
    async _requestWithETagCache(requestUrl, options) {
        const cacheEntry = this._eTagCache[requestUrl];
        const cachedETag = cacheEntry === null || cacheEntry === void 0 ? void 0 : cacheEntry.etag;
        const updatedOptions = cachedETag
            ? Object.assign(Object.assign({}, options), { headers: Object.assign(Object.assign({}, options.headers), { 'if-none-match': cachedETag }) }) : options;
        const res = await this._requests.fetch(requestUrl, updatedOptions);
        if (res.status === 304 && cacheEntry) {
            return { res, body: cacheEntry.body };
        }
        const etag = res.headers.get('etag');
        const body = await res.text();
        if (etag) {
            this._eTagCache[requestUrl] = { etag, body };
        }
        return { res, body };
    }
    async requestAllData(cb, queryParams = []) {
        var _a, _b;
        const options = {
            method: 'GET',
            headers: this._headers,
        };
        const uri = (0, js_sdk_common_1.getPollingUri)(this._serviceEndpoints, this._path, queryParams);
        (_a = this._logger) === null || _a === void 0 ? void 0 : _a.debug(`Requestor making request to uri: ${uri}`);
        try {
            const { res, body } = await this._requestWithETagCache(uri, options);
            (_b = this._logger) === null || _b === void 0 ? void 0 : _b.debug(`Requestor got (possibly cached) body: ${JSON.stringify(body)}`);
            if (res.headers.get(`x-ld-fd-fallback`) === `true`) {
                const err = new js_sdk_common_1.LDFlagDeliveryFallbackError(js_sdk_common_1.DataSourceErrorKind.ErrorResponse, `Response header indicates to fallback to FDv1.`, res.status);
                return cb(err, undefined, undefined);
            }
            if (res.status !== 200 && res.status !== 304) {
                const err = new js_sdk_common_1.LDPollingError(js_sdk_common_1.DataSourceErrorKind.ErrorResponse, `Unexpected status code: ${res.status}`, res.status);
                return cb(err, undefined, undefined);
            }
            return cb(undefined, res.status === 304 ? null : body, Object.fromEntries(res.headers.entries()));
        }
        catch (err) {
            return cb(err, undefined, undefined);
        }
    }
}
exports.default = Requestor;
//# sourceMappingURL=Requestor.js.map