"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const js_sdk_common_1 = require("@launchdarkly/js-sdk-common");
const booleanVariation_1 = require("./booleanVariation");
/**
 * A builder for feature flag rules to be used with {@link TestDataFlagBuilder}.
 *
 * In the LaunchDarkly model, a flag can have any number of rules, and
 * a rule can have any number of clauses. A clause is an individual test
 * such as "name is 'X'". A rule matches a user if all of the rule's
 * clauses match the user.
 *
 * To start defining a rule, use one of the flag builder's matching methods
 * such as `ifMatch`. This defines the first clause for the rule. Optionally,
 * you may add more clauses with the rule builder's methods such as `andMatch`.
 * Finally, call `thenReturn` to finish defining the rule.
 */
class TestDataRuleBuilder {
    /**
     * @internal
     */
    constructor(_flagBuilder, clauses, variation) {
        this._flagBuilder = _flagBuilder;
        this._clauses = [];
        if (clauses) {
            this._clauses = [...clauses];
        }
        if (variation !== undefined) {
            this._variation = variation;
        }
    }
    /**
     * Adds another clause using the "is one of" operator.
     *
     * For example, this creates a rule that returns `true` if the name is
     * "Patsy" and the country is "gb":
     *
     *     testData.flag('flag')
     *             .ifMatch('name', 'Patsy')
     *             .andMatch('country', 'gb')
     *             .thenReturn(true)
     *
     * @param contextKind the kind of the context
     * @param attribute the user attribute to match against
     * @param values values to compare to
     * @return the flag rule builder
     */
    andMatch(contextKind, attribute, ...values) {
        this._clauses.push({
            contextKind,
            attribute,
            attributeReference: new js_sdk_common_1.AttributeReference(attribute),
            op: 'in',
            values,
            negate: false,
        });
        return this;
    }
    /**
     * Adds another clause using the "is not one of" operator.
     *
     * For example, this creates a rule that returns `true` if the name is
     * "Patsy" and the country is not "gb":
     *
     *     testData.flag('flag')
     *             .ifMatch('name', 'Patsy')
     *             .andNotMatch('country', 'gb')
     *             .thenReturn(true)
     *
     * @param contextKind the kind of the context
     * @param attribute the user attribute to match against
     * @param values values to compare to
     * @return the flag rule builder
     */
    andNotMatch(contextKind, attribute, ...values) {
        this._clauses.push({
            contextKind,
            attribute,
            attributeReference: new js_sdk_common_1.AttributeReference(attribute),
            op: 'in',
            values,
            negate: true,
        });
        return this;
    }
    /**
     * Finishes defining the rule, specifying the result value as either a boolean or an index
     *
     * If the variation is a boolean value and the flag was not already a boolean
     * flag, this also changes it to be a boolean flag.
     *
     * If the variation is an integer, it specifies a variation out of whatever
     * variation values have already been defined.
     *
     * @param variation
     *    either `true` or `false` or the index of the desired variation:
     *    0 for the first, 1 for the second, etc.
     * @return the flag rule builder
     */
    thenReturn(variation) {
        if (js_sdk_common_1.TypeValidators.Boolean.is(variation)) {
            this._flagBuilder.booleanFlag();
            return this.thenReturn((0, booleanVariation_1.variationForBoolean)(variation));
        }
        this._variation = variation;
        this._flagBuilder.addRule(this);
        return this._flagBuilder;
    }
    /**
     * @internal
     */
    build(id) {
        return {
            id: `rule${id}`,
            variation: this._variation,
            clauses: this._clauses,
        };
    }
    /**
     * @internal
     */
    clone() {
        return new TestDataRuleBuilder(this._flagBuilder, this._clauses, this._variation);
    }
}
exports.default = TestDataRuleBuilder;
//# sourceMappingURL=TestDataRuleBuilder.js.map