"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
class InMemoryFeatureStore {
    constructor() {
        this._allData = {};
        this._initCalled = false;
    }
    get(kind, key, callback) {
        const items = this._allData[kind.namespace];
        if (items) {
            if (Object.prototype.hasOwnProperty.call(items, key)) {
                const item = items[key];
                if (item && !item.deleted) {
                    return callback === null || callback === void 0 ? void 0 : callback(item);
                }
            }
        }
        return callback === null || callback === void 0 ? void 0 : callback(null);
    }
    all(kind, callback) {
        var _a;
        const result = {};
        const items = (_a = this._allData[kind.namespace]) !== null && _a !== void 0 ? _a : {};
        Object.entries(items).forEach(([key, item]) => {
            if (item && !item.deleted) {
                result[key] = item;
            }
        });
        callback === null || callback === void 0 ? void 0 : callback(result);
    }
    init(allData, callback, initMetadata) {
        this.applyChanges(true, allData, callback, initMetadata);
    }
    delete(kind, key, version, callback) {
        const item = { key, version, deleted: true };
        this.applyChanges(false, {
            [kind.namespace]: {
                [key]: item,
            },
        }, callback);
    }
    upsert(kind, data, callback) {
        this.applyChanges(false, {
            [kind.namespace]: {
                [data.key]: data,
            },
        }, callback);
    }
    applyChanges(basis, data, callback, initMetadata, selector) {
        if (basis) {
            this._initCalled = true;
            this._allData = data;
            this._initMetadata = initMetadata;
        }
        else {
            const tempData = {};
            // shallow copy to protect against concurrent read
            Object.entries(this._allData).forEach(([namespace, items]) => {
                tempData[namespace] = Object.assign({}, items);
            });
            Object.entries(data).forEach(([namespace, items]) => {
                Object.keys(items || {}).forEach((key) => {
                    let existingItems = tempData[namespace];
                    if (!existingItems) {
                        existingItems = {};
                        tempData[namespace] = existingItems;
                    }
                    const item = items[key];
                    if (Object.hasOwnProperty.call(existingItems, key)) {
                        const old = existingItems[key];
                        // TODO: SDK-1046 - Determine if version check should be removed
                        if (!old || old.version < item.version) {
                            existingItems[key] = Object.assign({ key }, item);
                        }
                    }
                    else {
                        existingItems[key] = Object.assign({ key }, item);
                    }
                });
            });
            this._allData = tempData;
        }
        this._selector = selector;
        callback === null || callback === void 0 ? void 0 : callback();
    }
    initialized(callback) {
        return callback === null || callback === void 0 ? void 0 : callback(this._initCalled);
    }
    /* eslint-disable class-methods-use-this */
    close() {
        // For the memory store this is a no-op.
    }
    getDescription() {
        return 'memory';
    }
    getInitMetaData() {
        return this._initMetadata;
    }
    getSelector() {
        return this._selector;
    }
}
exports.default = InMemoryFeatureStore;
//# sourceMappingURL=InMemoryFeatureStore.js.map