"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createChartStore = void 0;
const toolkit_1 = require("@reduxjs/toolkit");
const chart_1 = require("./actions/chart");
const chart_settings_1 = require("./actions/chart_settings");
const colors_1 = require("./actions/colors");
const events_1 = require("./actions/events");
const specs_1 = require("./actions/specs");
const z_index_1 = require("./actions/z_index");
const chart_type_from_specs_1 = require("./chart_type_from_specs");
const get_initial_state_1 = require("./get_initial_state");
const interactions_1 = require("./reducers/interactions");
const get_initial_pointer_state_1 = require("./utils/get_initial_pointer_state");
const get_initial_tooltip_state_1 = require("./utils/get_initial_tooltip_state");
const default_settings_spec_1 = require("../specs/default_settings_spec");
const fast_deep_equal_1 = require("../utils/fast_deep_equal");
const handleChartActions = (builder) => {
    builder.addCase(chart_1.onChartRendered, (state) => {
        const chartRenderedCount = state.chartRendered ? state.chartRenderedCount : state.chartRenderedCount + 1;
        state.chartRendered = true;
        state.chartRenderedCount = chartRenderedCount;
    });
};
const handleChartSettingsActions = (builder) => {
    builder
        .addCase(chart_settings_1.updateParentDimensions, (state, action) => {
        if ((0, fast_deep_equal_1.deepEqual)(action.payload, state.parentDimensions)) {
            return;
        }
        state.interactions.prevDrilldown = state.interactions.drilldown;
        state.interactions.tooltip = (0, get_initial_tooltip_state_1.getInitialTooltipState)();
        state.interactions.pointer.pinned = null;
        state.parentDimensions = action.payload;
        state.chartRendered = false;
    })
        .addCase(chart_settings_1.updateChartTitles, (state, action) => {
        state.title = action.payload.title;
        state.description = action.payload.description;
    });
};
const handleColorsActions = (builder) => {
    builder
        .addCase(colors_1.clearTemporaryColors, (state) => {
        state.colors.temporary = {};
    })
        .addCase(colors_1.setTemporaryColor, (state, action) => {
        state.colors.temporary = {
            ...state.colors.temporary,
            ...action.payload.keys.reduce((acc, curr) => {
                acc[curr] = action.payload.color;
                return acc;
            }, {}),
        };
    })
        .addCase(colors_1.setPersistedColor, (state, action) => {
        state.colors.persisted = action.payload.keys.reduce((acc, curr) => {
            if (action.payload.color) {
                acc[curr] = action.payload.color;
            }
            else {
                delete acc[curr];
            }
            return acc;
        }, state.colors.persisted);
    });
};
const handleSpecsActions = (builder) => {
    builder
        .addCase(specs_1.upsertSpec, (state, action) => {
        if (state.specParsing) {
            state.specs[action.payload.id] = action.payload;
        }
        else {
            state.specs = { [default_settings_spec_1.DEFAULT_SETTINGS_SPEC.id]: default_settings_spec_1.DEFAULT_SETTINGS_SPEC, [action.payload.id]: action.payload };
        }
        state.specsInitialized = false;
        state.chartRendered = false;
        state.specParsing = true;
    })
        .addCase(specs_1.removeSpec, (state, action) => {
        const { [action.payload]: specToRemove, ...rest } = state.specs;
        state.specsInitialized = false;
        state.chartRendered = false;
        state.specParsing = false;
        state.specs = rest;
    })
        .addCase(specs_1.specParsed, (state) => {
        state.specsInitialized = true;
        state.specParsing = false;
        const newChartType = (0, chart_type_from_specs_1.chartTypeFromSpecs)(state.specs);
        if (state.chartType === newChartType)
            return;
        state.chartType = newChartType;
    })
        .addCase(specs_1.specUnmounted, (state) => {
        state.specsInitialized = false;
        state.chartRendered = false;
    });
};
const handleZIndexActions = (builder) => {
    builder.addCase(z_index_1.onComputedZIndex, (state, action) => {
        state.zIndex = action.payload;
    });
};
const handleEventsActions = (builder) => {
    builder.addCase(events_1.onExternalPointerEvent, (state, action) => {
        if (action.payload.chartId === state.chartId) {
            state.externalEvents.pointer = null;
            return;
        }
        state.externalEvents.pointer = action.payload;
        state.interactions.pointer = (0, get_initial_pointer_state_1.getInitialPointerState)();
        state.interactions.tooltip = (0, get_initial_tooltip_state_1.getInitialTooltipState)();
    });
};
const createChartSlice = (initialState) => (0, toolkit_1.createSlice)({
    name: 'chart',
    initialState,
    reducers: {},
    extraReducers: (builder) => {
        handleChartActions(builder);
        handleChartSettingsActions(builder);
        handleColorsActions(builder);
        handleEventsActions(builder);
        handleSpecsActions(builder);
        handleZIndexActions(builder);
        (0, interactions_1.handleKeyActions)(builder);
        (0, interactions_1.handleMouseActions)(builder);
        (0, interactions_1.handleLegendActions)(builder);
        (0, interactions_1.handleDOMElementActions)(builder);
        (0, interactions_1.handleTooltipActions)(builder);
    },
});
const createChartStore = (chartId, title, description) => {
    const initialState = (0, get_initial_state_1.getInitialState)(chartId, title, description);
    const chartSlice = createChartSlice(initialState);
    return (0, toolkit_1.configureStore)({
        reducer: chartSlice.reducer,
        middleware: (getDefaultMiddleware) => getDefaultMiddleware({
            serializableCheck: false,
        }),
        devTools: {
            name: `@elastic/charts - ${chartId}`,
        },
    });
};
exports.createChartStore = createChartStore;
//# sourceMappingURL=chart_state.js.map