"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createWrappedScopedClusterClientFactory = createWrappedScopedClusterClientFactory;
function _classPrivateFieldInitSpec(e, t, a) { _checkPrivateRedeclaration(e, t), t.set(e, a); }
function _checkPrivateRedeclaration(e, t) { if (t.has(e)) throw new TypeError("Cannot initialize the same private elements twice on an object"); }
function _classPrivateFieldSet(s, a, r) { return s.set(_assertClassBrand(s, a), r), r; }
function _classPrivateFieldGet(s, a) { return s.get(_assertClassBrand(s, a)); }
function _assertClassBrand(e, t, n) { if ("function" == typeof e ? e === t : e.has(t)) return arguments.length < 3 ? t : n; throw new TypeError("Private element is not present on this object"); }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createWrappedScopedClusterClientFactory(opts) {
  let numSearches = 0;
  let esSearchDurationMs = 0;
  let totalSearchDurationMs = 0;
  function logMetrics(metrics) {
    numSearches++;
    esSearchDurationMs += metrics.esSearchDuration;
    totalSearchDurationMs += metrics.totalSearchDuration;
  }
  const wrappedClient = wrapScopedClusterClient({
    ...opts,
    logMetricsFn: logMetrics
  });
  return {
    client: () => wrappedClient,
    getMetrics: () => {
      return {
        esSearchDurationMs,
        totalSearchDurationMs,
        numSearches
      };
    }
  };
}
var _asInternalUser = /*#__PURE__*/new WeakMap();
var _asCurrentUser = /*#__PURE__*/new WeakMap();
var _asSecondaryAuthUser = /*#__PURE__*/new WeakMap();
class WrappedScopedClusterClientImpl {
  constructor(opts) {
    _classPrivateFieldInitSpec(this, _asInternalUser, void 0);
    _classPrivateFieldInitSpec(this, _asCurrentUser, void 0);
    _classPrivateFieldInitSpec(this, _asSecondaryAuthUser, void 0);
    this.opts = opts;
  }
  get asInternalUser() {
    if (_classPrivateFieldGet(_asInternalUser, this) === undefined) {
      const {
        scopedClusterClient,
        ...rest
      } = this.opts;
      _classPrivateFieldSet(_asInternalUser, this, wrapEsClient({
        ...rest,
        esClient: scopedClusterClient.asInternalUser
      }));
    }
    return _classPrivateFieldGet(_asInternalUser, this);
  }
  get asCurrentUser() {
    if (_classPrivateFieldGet(_asCurrentUser, this) === undefined) {
      const {
        scopedClusterClient,
        ...rest
      } = this.opts;
      _classPrivateFieldSet(_asCurrentUser, this, wrapEsClient({
        ...rest,
        esClient: scopedClusterClient.asCurrentUser
      }));
    }
    return _classPrivateFieldGet(_asCurrentUser, this);
  }
  get asSecondaryAuthUser() {
    if (_classPrivateFieldGet(_asSecondaryAuthUser, this) === undefined) {
      const {
        scopedClusterClient,
        ...rest
      } = this.opts;
      _classPrivateFieldSet(_asSecondaryAuthUser, this, wrapEsClient({
        ...rest,
        esClient: scopedClusterClient.asSecondaryAuthUser
      }));
    }
    return _classPrivateFieldGet(_asSecondaryAuthUser, this);
  }
}
function wrapScopedClusterClient(opts) {
  return new WrappedScopedClusterClientImpl(opts);
}
function wrapEsClient(opts) {
  const {
    esClient,
    ...rest
  } = opts;
  const wrappedClient = esClient.child({});

  // Mutating the functions we want to wrap
  wrappedClient.transport.request = getWrappedTransportRequestFn({
    esClient: wrappedClient,
    ...rest
  });
  wrappedClient.search = getWrappedSearchFn({
    esClient: wrappedClient,
    ...rest
  });
  wrappedClient.eql.search = getWrappedEqlSearchFn({
    esClient: wrappedClient,
    ...rest
  });
  return wrappedClient;
}
function getWrappedTransportRequestFn(opts) {
  const originalRequestFn = opts.esClient.transport.request;
  const requestTimeout = opts.requestTimeout;

  // A bunch of overloads to make TypeScript happy

  async function request(params, options) {
    // Wrap ES|QL requests with an abort signal
    if (params.method === 'POST' && ['/_query', '/_query/async'].includes(params.path) || params.method === 'GET' && params.path.startsWith('/_query/async')) {
      let requestOptions = {};
      try {
        requestOptions = options !== null && options !== void 0 ? options : {};
        const start = Date.now();
        opts.logger.debug(() => `executing ES|QL query for rule ${opts.rule.alertTypeId}:${opts.rule.id} in space ${opts.rule.spaceId} - ${JSON.stringify(params)} - with options ${JSON.stringify(requestOptions)}${requestTimeout ? ` and ${requestTimeout}ms requestTimeout` : ''}`);
        const result = await originalRequestFn.call(opts.esClient.transport, params, {
          ...requestOptions,
          ...(requestTimeout ? {
            requestTimeout
          } : {}),
          signal: opts.abortController.signal
        });
        const end = Date.now();
        const durationMs = end - start;
        opts.logMetricsFn({
          esSearchDuration: 0,
          totalSearchDuration: durationMs
        });
        return result;
      } catch (e) {
        if (opts.abortController.signal.aborted) {
          throw new Error('ES|QL search has been aborted due to cancelled execution');
        }
        opts.logger.warn(`executing ES|QL query for rule ${opts.rule.alertTypeId}:${opts.rule.id} in space ${opts.rule.spaceId} - ${JSON.stringify(params)} - with options ${JSON.stringify(requestOptions)}${requestTimeout ? ` and ${requestTimeout}ms requestTimeout` : ''}`);
        throw e;
      }
    }

    // No wrap
    return await originalRequestFn.call(opts.esClient.transport, params, {
      ...options,
      ...(requestTimeout ? {
        requestTimeout
      } : {})
    });
  }
  return request;
}
function getWrappedEqlSearchFn(opts) {
  const originalEqlSearch = opts.esClient.eql.search;
  const requestTimeout = opts.requestTimeout;

  // A bunch of overloads to make TypeScript happy

  async function search(params, options) {
    let searchOptions = {};
    try {
      var _took;
      searchOptions = options !== null && options !== void 0 ? options : {};
      const start = Date.now();
      opts.logger.debug(() => `executing eql query for rule ${opts.rule.alertTypeId}:${opts.rule.id} in space ${opts.rule.spaceId} - ${JSON.stringify(params)} - with options ${JSON.stringify(searchOptions)}${requestTimeout ? ` and ${requestTimeout}ms requestTimeout` : ''}`);
      const result = await originalEqlSearch.call(opts.esClient.eql, params, {
        ...searchOptions,
        ...(requestTimeout ? {
          requestTimeout
        } : {}),
        signal: opts.abortController.signal
      });
      const end = Date.now();
      const durationMs = end - start;
      let took = 0;
      if (searchOptions.meta) {
        // when meta: true, response is TransportResult<EqlSearchResponse<TEvent>, unknown>
        took = result.body.took;
      } else {
        // when meta: false, response is EqlSearchResponse<TEvent>
        took = result.took;
      }
      opts.logMetricsFn({
        esSearchDuration: (_took = took) !== null && _took !== void 0 ? _took : 0,
        totalSearchDuration: durationMs
      });
      return result;
    } catch (e) {
      if (opts.abortController.signal.aborted) {
        throw new Error('EQL search has been aborted due to cancelled execution');
      }
      opts.logger.warn(`executing eql query for rule ${opts.rule.alertTypeId}:${opts.rule.id} in space ${opts.rule.spaceId} - ${JSON.stringify(params)} - with options ${JSON.stringify(searchOptions)}${requestTimeout ? ` and ${requestTimeout}ms requestTimeout` : ''}`);
      throw e;
    }
  }
  return search;
}
function getWrappedSearchFn(opts) {
  const originalSearch = opts.esClient.search;
  const requestTimeout = opts.requestTimeout;

  // A bunch of overloads to make TypeScript happy

  async function search(params, options) {
    let searchOptions = {};
    try {
      var _body$took, _body;
      searchOptions = options !== null && options !== void 0 ? options : {};
      const start = Date.now();
      opts.logger.debug(() => `executing query for rule ${opts.rule.alertTypeId}:${opts.rule.id} in space ${opts.rule.spaceId} - ${JSON.stringify(params)} - with options ${JSON.stringify(searchOptions)}${requestTimeout ? ` and ${requestTimeout}ms requestTimeout` : ''}`);
      const result = await originalSearch.call(opts.esClient, params, {
        ...searchOptions,
        ...(requestTimeout ? {
          requestTimeout
        } : {}),
        signal: opts.abortController.signal
      });
      const end = Date.now();
      const durationMs = end - start;
      let body;
      if (searchOptions.meta) {
        // when meta: true, response is TransportResult<SearchResponse<TDocument, TAggregations>, unknown>
        body = result.body;
      } else {
        // when meta: false, response is SearchResponse<TDocument, TAggregations>
        body = result;
      }
      opts.logMetricsFn({
        esSearchDuration: (_body$took = (_body = body) === null || _body === void 0 ? void 0 : _body.took) !== null && _body$took !== void 0 ? _body$took : 0,
        totalSearchDuration: durationMs
      });
      opts.logger.trace(() => `result of executing query for rule ${opts.rule.alertTypeId}:${opts.rule.id} in space ${opts.rule.spaceId}: ${JSON.stringify(body)}`);
      return result;
    } catch (e) {
      if (opts.abortController.signal.aborted) {
        throw new Error('Search has been aborted due to cancelled execution');
      }
      opts.logger.warn(`executing query for rule ${opts.rule.alertTypeId}:${opts.rule.id} in space ${opts.rule.spaceId} - ${JSON.stringify(params)} - with options ${JSON.stringify(searchOptions)}${requestTimeout ? ` and ${requestTimeout}ms requestTimeout` : ''}`);
      throw e;
    }
  }
  return search;
}