"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTraceSampleIds = getTraceSampleIds;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _lodash = require("lodash");
var _server = require("@kbn/observability-plugin/server");
var _common = require("@kbn/observability-plugin/common");
var _as_mutable_array = require("../../../common/utils/as_mutable_array");
var _apm = require("../../../common/es_fields/apm");
var _service_map = require("../../../common/service_map");
var _environment_query = require("../../../common/utils/environment_query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getTraceSampleIds({
  serviceName,
  environment,
  config,
  apmEventClient,
  start,
  end,
  serviceGroupKuery,
  kuery
}) {
  const query = {
    bool: {
      filter: [...(0, _server.rangeQuery)(start, end), ...(0, _environment_query.environmentQuery)(environment), ...(0, _server.kqlQuery)(serviceGroupKuery), ...(0, _server.kqlQuery)(kuery), ...(0, _server.termQuery)(_apm.SERVICE_NAME, serviceName)]
    }
  };
  const isUnfilteredGlobalServiceMap = !serviceName && !serviceGroupKuery && !kuery;
  const fingerprintBucketSize = isUnfilteredGlobalServiceMap ? config.serviceMapFingerprintGlobalBucketSize : config.serviceMapFingerprintBucketSize;
  const traceIdBucketSize = isUnfilteredGlobalServiceMap ? config.serviceMapTraceIdGlobalBucketSize : config.serviceMapTraceIdBucketSize;
  const samplerShardSize = traceIdBucketSize * 10;
  try {
    const searches = getQueries({
      query,
      samplerShardSize,
      fingerprintBucketSize,
      isUnfilteredGlobalServiceMap,
      traceIdBucketSize
    });
    const tracesSampleResponse = (await apmEventClient.msearch('get_trace_sample_ids', ...(isUnfilteredGlobalServiceMap ? [searches.traceSampleIds, searches.traceSampleIdsForSpanLinks] : [searches.traceSampleIds]))).responses;
    const samples = tracesSampleResponse.flatMap(response => {
      var _response$aggregation, _response$aggregation2;
      return (_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.connections.buckets.map(p => p.sample)) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
    });
    // make sure at least one trace per composite/connection bucket is queried
    const traceIdsWithPriority = samples.flatMap(sample => sample.trace_ids.buckets.map((sampleDocBucket, index) => ({
      traceId: sampleDocBucket.key,
      priority: index
    })));
    const traceIds = (0, _lodash.take)((0, _lodash.uniq)((0, _lodash.sortBy)(traceIdsWithPriority, 'priority').map(({
      traceId
    }) => traceId)), config.serviceMapMaxTraces);
    return {
      traceIds
    };
  } catch (error) {
    if ('displayName' in error && error.displayName === 'RequestTimeout') {
      throw _boom.default.internal(_service_map.SERVICE_MAP_TIMEOUT_ERROR);
    }
    throw error;
  }
}
function getQueries({
  query,
  fingerprintBucketSize,
  samplerShardSize,
  traceIdBucketSize,
  isUnfilteredGlobalServiceMap
}) {
  return {
    traceSampleIds: getFetchTraceSampleIdsParams({
      fingerprintBucketSize,
      isUnfilteredGlobalServiceMap,
      query,
      samplerShardSize,
      traceIdBucketSize
    }),
    traceSampleIdsForSpanLinks: getSampleTraceIdsForSpanLinksParams({
      query,
      fingerprintBucketSize,
      samplerShardSize,
      traceIdBucketSize
    })
  };
}
function getFetchTraceSampleIdsParams({
  query,
  fingerprintBucketSize,
  samplerShardSize,
  traceIdBucketSize,
  isUnfilteredGlobalServiceMap
}) {
  return {
    apm: {
      // perf optimization that is only possible on the global service map with no filters
      events: isUnfilteredGlobalServiceMap ? [_common.ProcessorEvent.span] : [_common.ProcessorEvent.span, _common.ProcessorEvent.transaction]
    },
    track_total_hits: false,
    size: 0,
    query: {
      bool: {
        filter: [...query.bool.filter, ...(isUnfilteredGlobalServiceMap ? (0, _server.existsQuery)(_apm.SPAN_DESTINATION_SERVICE_RESOURCE) : [])]
      }
    },
    aggs: {
      connections: {
        composite: {
          sources: (0, _as_mutable_array.asMutableArray)([{
            [_apm.SPAN_DESTINATION_SERVICE_RESOURCE]: {
              terms: {
                field: _apm.SPAN_DESTINATION_SERVICE_RESOURCE,
                missing_bucket: true
              }
            }
          }, {
            [_apm.SERVICE_NAME]: {
              terms: {
                field: _apm.SERVICE_NAME
              }
            }
          }, {
            [_apm.SERVICE_ENVIRONMENT]: {
              terms: {
                field: _apm.SERVICE_ENVIRONMENT,
                missing_bucket: true
              }
            }
          }]),
          size: fingerprintBucketSize
        },
        aggs: getAggregation({
          samplerShardSize,
          traceIdBucketSize
        })
      }
    }
  };
}
function getSampleTraceIdsForSpanLinksParams({
  query,
  fingerprintBucketSize,
  samplerShardSize,
  traceIdBucketSize
}) {
  return {
    apm: {
      events: [_common.ProcessorEvent.span, _common.ProcessorEvent.transaction]
    },
    track_total_hits: false,
    size: 0,
    query: {
      bool: {
        filter: [...query.bool.filter, {
          bool: {
            should: [...(0, _server.existsQuery)(_apm.SPAN_LINKS), ...(0, _server.existsQuery)(_apm.OTEL_SPAN_LINKS_SPAN_ID)],
            minimum_should_match: 1
          }
        }]
      }
    },
    aggs: {
      connections: {
        composite: {
          sources: (0, _as_mutable_array.asMutableArray)([
          // span.name and transaction.name are the same in otel
          // elastic apm will have either span.name or transaction.name, depending on the event type
          {
            [_apm.SPAN_NAME]: {
              terms: {
                field: _apm.SPAN_NAME,
                missing_bucket: true
              }
            }
          }, {
            [_apm.TRANSACTION_NAME]: {
              terms: {
                field: _apm.TRANSACTION_NAME,
                missing_bucket: true
              }
            }
          }, {
            [_apm.SERVICE_NAME]: {
              terms: {
                field: _apm.SERVICE_NAME
              }
            }
          }, {
            [_apm.SERVICE_ENVIRONMENT]: {
              terms: {
                field: _apm.SERVICE_ENVIRONMENT,
                missing_bucket: true
              }
            }
          }]),
          size: fingerprintBucketSize
        },
        aggs: getAggregation({
          samplerShardSize,
          traceIdBucketSize
        })
      }
    }
  };
}
function getAggregation({
  samplerShardSize,
  traceIdBucketSize
}) {
  return {
    sample: {
      sampler: {
        shard_size: samplerShardSize
      },
      aggs: {
        trace_ids: {
          terms: {
            field: _apm.TRACE_ID,
            size: traceIdBucketSize,
            execution_hint: 'map',
            // remove bias towards large traces by sorting on trace.id
            // which will be random-esque
            order: {
              _key: 'desc'
            }
          }
        }
      }
    }
  };
}