"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleCSV = handleCSV;
var _util = require("../../util");
var _processors = require("../../util/processors");
var _unparseable_csv_error = require("../../lib/errors/unparseable_csv_error");
var _columns = require("./columns");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// We will only create the processor for the first MAX_CSV_COLUMNS columns.
const MAX_CSV_COLUMNS = 100;
function createCSVPipeline(prefix, columns, headerValues) {
  const prefixedColumns = (0, _columns.prefixColumns)(columns, prefix);
  const dropProcessors = [];
  if (headerValues.length !== 0) {
    const dropValues = columns.reduce((acc, column, index) => {
      const headerValue = headerValues[index];
      if (headerValue !== undefined) {
        acc[column] = headerValue;
      }
      return acc;
    }, {});
    const dropProcessor = (0, _processors.createDropProcessor)(dropValues, prefix, 'remove_csv_header', 'Remove the CSV header by comparing row values to the header row.');
    dropProcessors.push(dropProcessor);
  }
  return [(0, _processors.createCSVProcessor)('message', prefixedColumns), ...dropProcessors];
}

/**
 * Processes CSV log data by parsing, testing, and converting to JSON format.
 *
 * The process follows three stages:
 * 1. Initial parsing with temporary column names (column1, column2, etc.)
 * 2. Testing with actual pipeline using package.dataStream.columnName format
 * 3. Converting to JSON format for further processing
 *
 * Final column names are determined by combining LLM suggestions, header row parsing,
 * and temporary columns as fallback. Includes header row handling and CSV-to-JSON conversion.
 *
 * @param param0 - Object containing state (log samples, format info) and Elasticsearch client
 * @returns Promise with JSON samples, processors, and chain label
 * @throws UnparseableCSVFormatError if CSV parsing fails
 */
async function handleCSV({
  state,
  client
}) {
  const jsonKey = 'json';
  const packageName = state.packageName;
  const dataStreamName = state.dataStreamName;
  const samples = state.logSamples;
  const temporaryColumns = (0, _columns.generateColumnNames)(Math.min((0, _columns.upperBoundForColumnCount)(samples), MAX_CSV_COLUMNS));
  const temporaryPipeline = createCSVPipeline([jsonKey], temporaryColumns, []);
  const {
    pipelineResults: tempResults,
    errors: tempErrors
  } = await (0, _util.createJSONInput)(temporaryPipeline, samples, client);
  if (tempErrors.length > 0) {
    throw new _unparseable_csv_error.UnparseableCSVFormatError(tempErrors);
  }

  // Some basic information we'll need later
  const prefix = [packageName, dataStreamName];

  // What columns does the LLM suggest?
  const llmProvidedColumns = (state.samplesFormat.columns || []).map(_columns.toSafeColumnName);

  // What columns do we get by parsing the header row, if any exists?
  const headerColumns = [];
  const headerValues = [];
  const csvRows = tempResults.map(result => result[jsonKey]);
  if (state.samplesFormat.header) {
    const headerRow = csvRows[0];
    headerValues.push(...(0, _columns.valuesFromHeader)(temporaryColumns, headerRow));
    headerColumns.push(...(0, _columns.columnsFromHeader)(temporaryColumns, headerRow));
  }

  // Combine all that information into a single list of columns
  const columns = Array.from((0, _columns.yieldUniqueColumnNames)((0, _columns.totalColumnCount)(temporaryColumns, csvRows), [llmProvidedColumns, headerColumns], temporaryColumns));

  // These processors extract CSV fields into a specific key.
  const csvHandlingProcessors = createCSVPipeline(prefix, columns, headerValues);

  // Test the processors on the samples provided
  const {
    errors
  } = await (0, _util.createJSONInput)(csvHandlingProcessors, samples, client);
  if (errors.length > 0) {
    throw new _unparseable_csv_error.UnparseableCSVFormatError(errors);
  }

  // These processors extract CSV fields into a specific key.
  const csvToJSONProcessors = createCSVPipeline([jsonKey], columns, headerValues);
  const {
    pipelineResults: jsonResults,
    errors: jsonErrors
  } = await (0, _util.createJSONInput)(csvToJSONProcessors, samples, client);
  if (jsonErrors.length > 0) {
    throw new _unparseable_csv_error.UnparseableCSVFormatError(jsonErrors);
  }
  const jsonSamples = jsonResults.map(log => log[jsonKey]).map(log => JSON.stringify(log));
  return {
    jsonSamples,
    additionalProcessors: [...state.additionalProcessors, ...csvHandlingProcessors],
    lastExecutedChain: 'handleCSV'
  };
}