"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.resolveCaseRoute = exports.getCaseRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../../../common/constants");
var _error = require("../../../common/error");
var _create_cases_route = require("../create_cases_route");
var _constants2 = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const params = {
  params: _configSchema.schema.object({
    case_id: _configSchema.schema.string()
  })
};
const getCaseRoute = () => (0, _create_cases_route.createCasesRoute)({
  method: 'get',
  path: _constants.CASE_DETAILS_URL,
  security: _constants2.DEFAULT_CASES_ROUTE_SECURITY,
  params,
  routerOptions: {
    access: 'public',
    summary: `Get a case`,
    tags: ['oas-tag:cases']
  },
  handler: async ({
    context,
    request,
    response,
    logger,
    kibanaVersion
  }) => {
    try {
      const caseContext = await context.cases;
      const casesClient = await caseContext.getCasesClient();
      const id = request.params.case_id;
      const res = await casesClient.cases.get({
        id,
        includeComments: false
      });
      const {
        comments,
        ...caseWithoutComments
      } = res;
      return response.ok({
        body: caseWithoutComments
      });
    } catch (error) {
      throw (0, _error.createCaseError)({
        message: `Failed to retrieve case in route case id: ${request.params.case_id} \n${error}`,
        error
      });
    }
  }
});
exports.getCaseRoute = getCaseRoute;
const resolveCaseRoute = exports.resolveCaseRoute = (0, _create_cases_route.createCasesRoute)({
  method: 'get',
  path: `${_constants.CASE_DETAILS_URL}/resolve`,
  security: _constants2.DEFAULT_CASES_ROUTE_SECURITY,
  routerOptions: {
    access: 'internal'
  },
  params: {
    ...params,
    query: _configSchema.schema.object({
      /**
       * @deprecated since version 8.1.0
       */
      includeComments: _configSchema.schema.boolean({
        defaultValue: true,
        meta: {
          deprecated: true
        }
      })
    })
  },
  handler: async ({
    context,
    request,
    response
  }) => {
    try {
      const caseContext = await context.cases;
      const casesClient = await caseContext.getCasesClient();
      const id = request.params.case_id;
      const res = await casesClient.cases.resolve({
        id,
        includeComments: request.query.includeComments
      });
      return response.ok({
        body: res
      });
    } catch (error) {
      throw (0, _error.createCaseError)({
        message: `Failed to retrieve case in resolve route case id: ${request.params.case_id} \ninclude comments: ${request.query.includeComments}: ${error}`,
        error
      });
    }
  }
});