"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerFavoritesRoutes = registerFavoritesRoutes;
var _server = require("@kbn/core/server");
var _configSchema = require("@kbn/config-schema");
var _favorites_service = require("./favorites_service");
var _favorites_saved_object = require("./favorites_saved_object");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * @public
 * Response for get favorites API
 */

function registerFavoritesRoutes({
  core,
  logger,
  favoritesRegistry
}) {
  const typeSchema = _configSchema.schema.string({
    validate: type => {
      if (!favoritesRegistry.hasType(type)) {
        return `Unknown favorite type: ${type}`;
      }
    }
  });
  const metadataSchema = _configSchema.schema.maybe(_configSchema.schema.object({
    // validated later by the registry depending on the type
  }, {
    unknowns: 'allow'
  }));
  const router = core.http.createRouter();
  const getSavedObjectClient = coreRequestHandlerContext => {
    // We need to exclude security extension to access the `favorite` type which not every user has access to
    // and give access only to the current user's favorites through this API
    return coreRequestHandlerContext.savedObjects.getClient({
      includedHiddenTypes: [_favorites_saved_object.favoritesSavedObjectType.name],
      excludedExtensions: [_server.SECURITY_EXTENSION_ID]
    });
  };
  router.post({
    path: '/internal/content_management/favorites/{type}/{id}/favorite',
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string(),
        type: typeSchema
      }),
      body: _configSchema.schema.maybe(_configSchema.schema.nullable(_configSchema.schema.object({
        metadata: metadataSchema
      })))
    },
    security: {
      authz: {
        enabled: false,
        reason: "we don't protect the route with any access tags as we only give access to the current user's favorites ids"
      }
    }
  }, async (requestHandlerContext, request, response) => {
    var _coreRequestHandlerCo, _request$body;
    const coreRequestHandlerContext = await requestHandlerContext.core;
    const userId = (_coreRequestHandlerCo = coreRequestHandlerContext.security.authc.getCurrentUser()) === null || _coreRequestHandlerCo === void 0 ? void 0 : _coreRequestHandlerCo.profile_uid;
    if (!userId) {
      return response.forbidden();
    }
    const type = request.params.type;
    const favorites = new _favorites_service.FavoritesService(type, userId, {
      savedObjectClient: getSavedObjectClient(coreRequestHandlerContext),
      logger,
      favoritesRegistry
    });
    const id = request.params.id;
    const metadata = (_request$body = request.body) === null || _request$body === void 0 ? void 0 : _request$body.metadata;
    try {
      favoritesRegistry.validateMetadata(type, metadata);
    } catch (e) {
      return response.badRequest({
        body: {
          message: e.message
        }
      });
    }
    try {
      const favoritesResult = await favorites.addFavorite({
        id,
        metadata
      });
      const addFavoritesResponse = {
        favoriteIds: favoritesResult.favoriteIds
      };
      return response.ok({
        body: addFavoritesResponse
      });
    } catch (e) {
      if (e instanceof _favorites_service.FavoritesLimitExceededError) {
        return response.forbidden({
          body: {
            message: e.message
          }
        });
      }
      throw e; // unexpected error, let the global error handler deal with it
    }
  });
  router.post({
    path: '/internal/content_management/favorites/{type}/{id}/unfavorite',
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string(),
        type: typeSchema
      })
    },
    security: {
      authz: {
        enabled: false,
        reason: "we don't protect the route with any access tags as we only give access to the current user's favorites ids"
      }
    }
  }, async (requestHandlerContext, request, response) => {
    var _coreRequestHandlerCo2;
    const coreRequestHandlerContext = await requestHandlerContext.core;
    const userId = (_coreRequestHandlerCo2 = coreRequestHandlerContext.security.authc.getCurrentUser()) === null || _coreRequestHandlerCo2 === void 0 ? void 0 : _coreRequestHandlerCo2.profile_uid;
    if (!userId) {
      return response.forbidden();
    }
    const type = request.params.type;
    const favorites = new _favorites_service.FavoritesService(type, userId, {
      savedObjectClient: getSavedObjectClient(coreRequestHandlerContext),
      logger,
      favoritesRegistry
    });
    const favoritesResult = await favorites.removeFavorite({
      id: request.params.id
    });
    const removeFavoriteResponse = {
      favoriteIds: favoritesResult.favoriteIds
    };
    return response.ok({
      body: removeFavoriteResponse
    });
  });
  router.get({
    path: '/internal/content_management/favorites/{type}',
    validate: {
      params: _configSchema.schema.object({
        type: typeSchema
      })
    },
    security: {
      authz: {
        enabled: false,
        reason: "we don't protect the route with any access tags as we only give access to the current user's favorites ids"
      }
    }
  }, async (requestHandlerContext, request, response) => {
    var _coreRequestHandlerCo3;
    const coreRequestHandlerContext = await requestHandlerContext.core;
    const userId = (_coreRequestHandlerCo3 = coreRequestHandlerContext.security.authc.getCurrentUser()) === null || _coreRequestHandlerCo3 === void 0 ? void 0 : _coreRequestHandlerCo3.profile_uid;
    if (!userId) {
      return response.forbidden();
    }
    const type = request.params.type;
    const favorites = new _favorites_service.FavoritesService(type, userId, {
      savedObjectClient: getSavedObjectClient(coreRequestHandlerContext),
      logger,
      favoritesRegistry
    });
    const favoritesResult = await favorites.getFavorites();
    const favoritesResponse = {
      favoriteIds: favoritesResult.favoriteIds,
      favoriteMetadata: favoritesResult.favoriteMetadata
    };
    return response.ok({
      body: favoritesResponse
    });
  });
}