"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.reindex = void 0;
var Either = _interopRequireWildcard(require("fp-ts/Either"));
var Option = _interopRequireWildcard(require("fp-ts/Option"));
var _catch_retryable_es_client_errors = require("./catch_retryable_es_client_errors");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/** @internal */

/** @internal */

/**
 * Reindex documents from the `sourceIndex` into the `targetIndex`. Returns a
 * task ID which can be tracked for progress.
 *
 * @remarks This action is idempotent allowing several Kibana instances to run
 * this in parallel. By using `op_type: 'create', conflicts: 'proceed'` there
 * will be only one write per reindexed document.
 */
const reindex = ({
  client,
  sourceIndex,
  targetIndex,
  reindexScript,
  requireAlias,
  excludeOnUpgradeQuery,
  batchSize
}) => () => {
  return client.reindex({
    // Require targetIndex to be an alias. Prevents a new index from being
    // created if targetIndex doesn't exist.
    require_alias: requireAlias,
    // Ignore version conflicts from existing documents
    conflicts: 'proceed',
    source: {
      index: sourceIndex,
      // Set reindex batch size
      size: batchSize,
      // Exclude saved object types
      query: excludeOnUpgradeQuery
    },
    dest: {
      index: targetIndex,
      // Don't override existing documents, only create if missing
      op_type: 'create'
    },
    script: Option.fold(() => undefined, script => ({
      source: script,
      lang: 'painless'
    }))(reindexScript),
    // force a refresh so that we can query the target index
    refresh: true,
    // Create a task and return task id instead of blocking until complete
    wait_for_completion: false
  }).then(({
    task: taskId
  }) => {
    return Either.right({
      taskId: String(taskId)
    });
  }).catch(_catch_retryable_es_client_errors.catchRetryableEsClientErrors);
};
exports.reindex = reindex;