"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UiSettingsClientCommon = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _rxjs = require("rxjs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class UiSettingsClientCommon {
  constructor(params) {
    (0, _defineProperty2.default)(this, "update$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "updateErrors$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "api", void 0);
    (0, _defineProperty2.default)(this, "defaults", void 0);
    (0, _defineProperty2.default)(this, "cache", void 0);
    this.api = params.api;
    this.defaults = (0, _lodash.cloneDeep)(params.defaults);
    this.cache = (0, _lodash.defaultsDeep)(Object.create(null), this.defaults, (0, _lodash.cloneDeep)(params.initialSettings));
    params.done$.subscribe({
      complete: () => {
        this.update$.complete();
        this.updateErrors$.complete();
      }
    });
  }
  getAll() {
    return (0, _lodash.cloneDeep)(this.cache);
  }
  get(key, defaultOverride) {
    const declared = this.isDeclared(key);
    if (!declared && defaultOverride !== undefined) {
      return defaultOverride;
    }
    if (!declared) {
      throw new Error(`Unexpected \`IUiSettingsClient.get("${key}")\` call on unrecognized configuration setting "${key}".
Setting an initial value via \`IUiSettingsClient.set("${key}", value)\` before attempting to retrieve
any custom setting value for "${key}" may fix this issue.
You can use \`IUiSettingsClient.get("${key}", defaultValue)\`, which will just return
\`defaultValue\` when the key is unrecognized.`);
    }
    const type = this.cache[key].type;
    const userValue = this.cache[key].userValue;
    const defaultValue = defaultOverride !== undefined ? defaultOverride : this.cache[key].value;
    const value = userValue == null ? defaultValue : userValue;
    if (type === 'json') {
      return JSON.parse(value);
    }
    if (type === 'number') {
      return parseFloat(value);
    }
    return value;
  }
  get$(key, defaultOverride) {
    return (0, _rxjs.concat)((0, _rxjs.defer)(() => (0, _rxjs.of)(this.get(key, defaultOverride))), this.update$.pipe((0, _rxjs.filter)(update => update.key === key), (0, _rxjs.map)(() => this.get(key, defaultOverride))));
  }
  async set(key, value) {
    return await this.update(key, value);
  }
  async remove(key) {
    return await this.update(key, null);
  }
  isDeclared(key) {
    return (
      // @ts-expect-error
      (key !== '__proto__' || key !== 'constructor' || key !== 'prototype') && key in this.cache
    );
  }
  isDefault(key) {
    return !this.isDeclared(key) || !this.cache[key].userValue || this.cache[key].userValue === this.cache[key].value;
  }
  isCustom(key) {
    return this.isDeclared(key) && !('value' in this.cache[key]);
  }
  isOverridden(key) {
    return this.isDeclared(key) && Boolean(this.cache[key].isOverridden);
  }
  isStrictReadonly(key) {
    return this.isDeclared(key) && Boolean(this.cache[key].readonlyMode === 'strict');
  }
  getUpdate$() {
    return this.update$.asObservable();
  }
  getUpdateErrors$() {
    return this.updateErrors$.asObservable();
  }
  async validateValue(key, value) {
    try {
      const resp = await this.api.validate(key, value);
      const isValid = resp.valid;
      return isValid ? {
        successfulValidation: true,
        valid: true
      } : {
        successfulValidation: true,
        valid: false,
        errorMessage: resp.errorMessage
      };
    } catch (error) {
      this.updateErrors$.next(error);
      return {
        successfulValidation: false
      };
    }
  }
  assertUpdateAllowed(key) {
    if (this.isOverridden(key)) {
      throw new Error(`Unable to update "${key}" because its value is overridden by the Kibana server`);
    }
    if (this.isStrictReadonly(key)) {
      throw new Error(`Unable to update "${key}" because this setting is not in the allowlist.`);
    }
  }
  setLocally(key, newValue) {
    this.assertUpdateAllowed(key);
    if (!this.isDeclared(key)) {
      this.cache[key] = {};
    }
    const oldValue = this.get(key);
    if (newValue === null) {
      delete this.cache[key].userValue;
    } else {
      const {
        type
      } = this.cache[key];
      if (type === 'json' && typeof newValue !== 'string') {
        this.cache[key].userValue = JSON.stringify(newValue);
      } else {
        this.cache[key].userValue = newValue;
      }
    }
    this.update$.next({
      key,
      newValue,
      oldValue
    });
  }
}
exports.UiSettingsClientCommon = UiSettingsClientCommon;