"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findDashboardById = findDashboardById;
exports.findDashboardIdByTitle = findDashboardIdByTitle;
exports.findDashboardsByIds = findDashboardsByIds;
exports.searchDashboards = searchDashboards;
var _ = require("..");
var _telemetry_constants = require("../../../utils/telemetry_constants");
var _kibana_services = require("../../kibana_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

async function searchDashboards({
  hasNoReference,
  hasReference,
  options,
  search,
  size
}) {
  const {
    hits,
    pagination: {
      total
    }
  } = await _kibana_services.contentManagementService.client.search({
    contentTypeId: _telemetry_constants.DASHBOARD_CONTENT_ID,
    query: {
      text: search ? `${search}*` : undefined,
      limit: size,
      tags: {
        included: (hasReference !== null && hasReference !== void 0 ? hasReference : []).map(({
          id
        }) => id),
        excluded: (hasNoReference !== null && hasNoReference !== void 0 ? hasNoReference : []).map(({
          id
        }) => id)
      }
    },
    options
  });
  return {
    total,
    hits
  };
}
async function findDashboardById(id) {
  const dashboardContentManagementCache = (0, _.getDashboardContentManagementCache)();

  /** If the dashboard exists in the cache, then return the result from that */
  const cachedDashboard = dashboardContentManagementCache.fetchDashboard(id);
  if (cachedDashboard) {
    return {
      id,
      status: 'success',
      attributes: cachedDashboard.item.attributes,
      references: cachedDashboard.item.references
    };
  }

  /** Otherwise, fetch the dashboard from the content management client, add it to the cache, and return the result */
  try {
    var _response$item$refere;
    const response = await _kibana_services.contentManagementService.client.get({
      contentTypeId: _telemetry_constants.DASHBOARD_CONTENT_ID,
      id
    });
    if ('error' in response.item) {
      throw response.item.error;
    }
    dashboardContentManagementCache.addDashboard(response);
    return {
      id,
      status: 'success',
      attributes: response.item.attributes,
      references: (_response$item$refere = response.item.references) !== null && _response$item$refere !== void 0 ? _response$item$refere : []
    };
  } catch (e) {
    return {
      status: 'error',
      error: e.body || e.message,
      id
    };
  }
}
async function findDashboardsByIds(ids) {
  const findPromises = ids.map(id => findDashboardById(id));
  const results = await Promise.all(findPromises);
  return results;
}
async function findDashboardIdByTitle(title) {
  const {
    hits
  } = await _kibana_services.contentManagementService.client.search({
    contentTypeId: _telemetry_constants.DASHBOARD_CONTENT_ID,
    query: {
      text: title ? `${title}*` : undefined,
      limit: 10
    },
    options: {
      onlyTitle: true
    }
  });

  // The search isn't an exact match, lets see if we can find a single exact match to use
  const matchingDashboards = hits.filter(hit => hit.attributes.title.toLowerCase() === title.toLowerCase());
  if (matchingDashboards.length === 1) {
    return {
      id: matchingDashboards[0].id
    };
  }
}