"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createEscapeValue = createEscapeValue;
var _constants = require("./constants");
var _formula_checks = require("./formula_checks");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// string with the delimiter/separator already inside need to be wrapped in quotes
// i.e. string with delimiter char in it like free text or some number formatting (1143 => 1,143)
function shouldBeQuoted(value, delimiter) {
  const trimmedSeparator = delimiter.trim();
  return value.includes(trimmedSeparator) && _constants.commonQuotedDelimiters.has(trimmedSeparator);
}

/**
 * Create a function that will escape CSV values like "=", "@" and "+" with a
 * "'". This will also place CSV values in "" if contain non-alphanumeric chars.
 *
 * For example:
 *
 * Given: =1+1
 * Returns: "'=1+1"
 *
 * See OWASP: https://www.owasp.org/index.php/CSV_Injection.
 */
function createEscapeValue({
  separator,
  quoteValues,
  escapeFormulaValues
}) {
  return function escapeValue(val) {
    if (val && typeof val === 'string') {
      const formulasEscaped = escapeFormulaValues && (0, _formula_checks.cellHasFormulas)(val) ? "'" + val : val;
      if (quoteValues) {
        if (_constants.nonAlphaNumRE.test(formulasEscaped)) {
          return `"${formulasEscaped.replace(_constants.allDoubleQuoteRE, '""')}"`;
        }
      }
    }
    // raw multi-terms are stringified as T1,T2,T3 so check if the final value contains the
    // csv separator before returning (usually for raw values)
    const stringVal = val == null ? '' : val.toString();
    return quoteValues && shouldBeQuoted(stringVal, separator) ? `"${stringVal}"` : stringVal;
  };
}