"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getIndexPatternTelemetry = getIndexPatternTelemetry;
exports.minMaxAvgLoC = void 0;
exports.registerIndexPatternsUsageCollector = registerIndexPatternsUsageCollector;
exports.updateMin = exports.updateMax = void 0;
var _server = require("@kbn/core/server");
var _common = require("../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const minMaxAvgLoC = scripts => {
  const lengths = scripts.map(script => (script === null || script === void 0 ? void 0 : script.split(/\r\n|\r|\n/).length) || 0).sort();
  return {
    min: lengths[0],
    max: lengths[lengths.length - 1],
    avg: lengths.reduce((col, count) => col + count, 0) / lengths.length
  };
};
exports.minMaxAvgLoC = minMaxAvgLoC;
const updateMin = (currentMin, newVal) => {
  if (currentMin === undefined || currentMin > newVal) {
    return newVal;
  } else {
    return currentMin;
  }
};
exports.updateMin = updateMin;
const updateMax = (currentMax, newVal) => {
  if (currentMax === undefined || currentMax < newVal) {
    return newVal;
  } else {
    return currentMax;
  }
};
exports.updateMax = updateMax;
async function getIndexPatternTelemetry(savedObjectsService) {
  const countSummaryDefaults = {
    min: undefined,
    max: undefined,
    avg: undefined
  };
  const results = {
    indexPatternsCount: 0,
    indexPatternsWithScriptedFieldCount: 0,
    indexPatternsWithRuntimeFieldCount: 0,
    scriptedFieldCount: 0,
    runtimeFieldCount: 0,
    perIndexPattern: {
      scriptedFieldCount: {
        ...countSummaryDefaults
      },
      runtimeFieldCount: {
        ...countSummaryDefaults
      },
      scriptedFieldLineCount: {
        ...countSummaryDefaults
      },
      runtimeFieldLineCount: {
        ...countSummaryDefaults
      }
    }
  };
  const findOptions = {
    type: _common.DATA_VIEW_SAVED_OBJECT_TYPE,
    perPage: 1000,
    fields: ['fields', 'runtimeFieldMap']
  };
  const finder = savedObjectsService.createPointInTimeFinder(findOptions);
  for await (const response of finder.find()) {
    const {
      saved_objects: savedObjects,
      total
    } = response;
    results.indexPatternsCount = total;
    savedObjects.forEach(obj => {
      var _obj$attributes, _obj$attributes2;
      const fields = (_obj$attributes = obj.attributes) !== null && _obj$attributes !== void 0 && _obj$attributes.fields ? JSON.parse(obj.attributes.fields) || [] : [];
      const runtimeFieldsMap = (_obj$attributes2 = obj.attributes) !== null && _obj$attributes2 !== void 0 && _obj$attributes2.runtimeFieldMap ? JSON.parse(obj.attributes.runtimeFieldMap) || {} : {};
      const scriptedFields = fields.filter(fld => !!fld.script);
      const runtimeFields = Object.values(runtimeFieldsMap);

      // calc LoC
      if (scriptedFields.length > 0) {
        // increment counts
        results.indexPatternsWithScriptedFieldCount++;
        results.scriptedFieldCount += scriptedFields.length;

        // calc LoC
        results.perIndexPattern.scriptedFieldLineCount = minMaxAvgLoC(scriptedFields.map(fld => fld.script || ''));

        // calc field counts
        results.perIndexPattern.scriptedFieldCount.min = updateMin(results.perIndexPattern.scriptedFieldCount.min, scriptedFields.length);
        results.perIndexPattern.scriptedFieldCount.max = updateMax(results.perIndexPattern.scriptedFieldCount.max, scriptedFields.length);
        results.perIndexPattern.scriptedFieldCount.avg = results.scriptedFieldCount / results.indexPatternsWithScriptedFieldCount;
      }
      if (runtimeFields.length > 0) {
        // increment counts
        results.indexPatternsWithRuntimeFieldCount++;
        results.runtimeFieldCount += runtimeFields.length;

        // calc LoC
        const runtimeFieldScripts = runtimeFields.map(fld => {
          var _fld$script;
          return ((_fld$script = fld.script) === null || _fld$script === void 0 ? void 0 : _fld$script.source) || '';
        });
        results.perIndexPattern.runtimeFieldLineCount = minMaxAvgLoC(runtimeFieldScripts);

        // calc field counts
        results.perIndexPattern.runtimeFieldCount.min = updateMin(results.perIndexPattern.runtimeFieldCount.min, runtimeFields.length);
        results.perIndexPattern.runtimeFieldCount.max = updateMax(results.perIndexPattern.runtimeFieldCount.max, runtimeFields.length);
        results.perIndexPattern.runtimeFieldCount.avg = results.runtimeFieldCount / results.indexPatternsWithRuntimeFieldCount;
      }
    });
  }
  return results;
}
function registerIndexPatternsUsageCollector(getStartServices, usageCollection) {
  if (!usageCollection) {
    return;
  }
  const indexPatternUsageCollector = usageCollection.makeUsageCollector({
    type: 'index-patterns',
    isReady: () => true,
    fetch: async () => {
      const [{
        savedObjects
      }] = await getStartServices();
      const savedObjectsService = new _server.SavedObjectsClient(savedObjects.createInternalRepository());
      return await getIndexPatternTelemetry(savedObjectsService);
    },
    schema: {
      indexPatternsCount: {
        type: 'long'
      },
      indexPatternsWithScriptedFieldCount: {
        type: 'long'
      },
      indexPatternsWithRuntimeFieldCount: {
        type: 'long'
      },
      scriptedFieldCount: {
        type: 'long'
      },
      runtimeFieldCount: {
        type: 'long'
      },
      perIndexPattern: {
        scriptedFieldCount: {
          min: {
            type: 'long'
          },
          max: {
            type: 'long'
          },
          avg: {
            type: 'float'
          }
        },
        runtimeFieldCount: {
          min: {
            type: 'long'
          },
          max: {
            type: 'long'
          },
          avg: {
            type: 'float'
          }
        },
        scriptedFieldLineCount: {
          min: {
            type: 'long'
          },
          max: {
            type: 'long'
          },
          avg: {
            type: 'float'
          }
        },
        runtimeFieldLineCount: {
          min: {
            type: 'long'
          },
          max: {
            type: 'long'
          },
          avg: {
            type: 'float'
          }
        }
      }
    }
  });
  usageCollection.registerCollector(indexPatternUsageCollector);
}