"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getNumericFieldsStatsRequest = exports.fetchNumericFieldsStats = void 0;
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _mlIsDefined = require("@kbn/ml-is-defined");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _utils = require("./utils");
var _build_random_sampler_agg = require("./build_random_sampler_agg");
var _constants = require("./constants");
var _process_distribution_data = require("../../utils/process_distribution_data");
var _field_stats = require("../../../../../common/types/field_stats");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getNumericFieldsStatsRequest = (params, fields) => {
  const {
    index,
    query,
    runtimeFieldMap
  } = params;
  const size = 0;

  // Build the percents parameter which defines the percentiles to query
  // for the metric distribution data.
  // Use a fixed percentile spacing of 5%.
  let count = 0;
  const percents = Array.from(Array(_constants.MAX_PERCENT / _constants.PERCENTILE_SPACING), () => count += _constants.PERCENTILE_SPACING);
  const aggs = {};
  fields.forEach((field, i) => {
    const {
      safeFieldName,
      supportedAggs
    } = field;
    const include = !(0, _mlIsDefined.isDefined)(supportedAggs);
    if (include || supportedAggs.has('stats')) {
      aggs[`${safeFieldName}_field_stats`] = {
        filter: {
          exists: {
            field: field.fieldName
          }
        },
        aggs: {
          actual_stats: {
            stats: {
              field: field.fieldName
            }
          }
        }
      };
    }
    if (include || supportedAggs.has('min') && supportedAggs.has('max')) {
      aggs[`${safeFieldName}_min_max`] = {
        filter: {
          exists: {
            field: field.fieldName
          }
        },
        aggs: {
          min: {
            min: {
              field: field.fieldName
            }
          },
          max: {
            max: {
              field: field.fieldName
            }
          }
        }
      };
    }
    if (include || supportedAggs.has('percentiles')) {
      aggs[`${safeFieldName}_percentiles`] = {
        percentiles: {
          field: field.fieldName,
          percents,
          keyed: false
        }
      };
    }
    if (include || supportedAggs.has('terms')) {
      const top = {
        terms: {
          field: field.fieldName,
          size: 10,
          order: {
            _count: 'desc'
          }
        }
      };
      aggs[`${safeFieldName}_top`] = top;
    }
  });
  const searchBody = {
    query,
    aggs: (0, _build_random_sampler_agg.buildAggregationWithSamplingOption)(aggs, params.samplingOption),
    ...((0, _mlIsPopulatedObject.isPopulatedObject)(runtimeFieldMap) ? {
      runtime_mappings: runtimeFieldMap
    } : {})
  };
  return {
    index,
    size,
    ...searchBody
  };
};
exports.getNumericFieldsStatsRequest = getNumericFieldsStatsRequest;
const processStats = (safeFieldName, aggregations, aggsPath) => {
  const fieldStatsResp = (0, _lodash.get)(aggregations, [...aggsPath, `${safeFieldName}_field_stats`, 'actual_stats'], undefined);
  if (fieldStatsResp) {
    return {
      min: (0, _lodash.get)(fieldStatsResp, 'min'),
      max: (0, _lodash.get)(fieldStatsResp, 'max'),
      avg: (0, _lodash.get)(fieldStatsResp, 'avg')
    };
  }
  const minMaxResp = (0, _lodash.get)(aggregations, [...aggsPath, `${safeFieldName}_min_max`], {});
  if (minMaxResp) {
    return {
      min: (0, _lodash.get)(minMaxResp, ['min', 'value']),
      max: (0, _lodash.get)(minMaxResp, ['max', 'value'])
    };
  }
  return {};
};
const fetchNumericFieldsStats = (dataSearch, params, fields, options) => {
  const request = getNumericFieldsStatsRequest(params, fields);
  return dataSearch.search({
    params: request
  }, options).pipe((0, _rxjs.catchError)(e => {
    return (0, _rxjs.of)({
      fields,
      error: (0, _mlErrorUtils.extractErrorProperties)(e)
    });
  }), (0, _rxjs.map)(resp => {
    if (!(0, _field_stats.isIKibanaSearchResponse)(resp)) return resp;
    const aggregations = resp.rawResponse.aggregations;
    const aggsPath = ['sample'];
    const batchStats = [];
    fields.forEach((field, i) => {
      const safeFieldName = field.safeFieldName;
      const docCount = (0, _lodash.get)(aggregations, [...aggsPath, `${safeFieldName}_field_stats`, 'doc_count'], 0);
      const topAggsPath = [...aggsPath, `${safeFieldName}_top`];
      const fieldAgg = (0, _lodash.get)(aggregations, [...topAggsPath], {});
      const {
        topValuesSampleSize,
        topValues
      } = (0, _utils.processTopValues)(fieldAgg);
      const stats = {
        fieldName: field.fieldName,
        ...processStats(safeFieldName, aggregations, aggsPath),
        isTopValuesSampled: (0, _field_stats.isNormalSamplingOption)(params.samplingOption) || (0, _mlIsDefined.isDefined)(params.samplingProbability) && params.samplingProbability < 1,
        topValues,
        topValuesSampleSize,
        topValuesSamplerShardSize: (0, _lodash.get)(aggregations, ['sample', 'doc_count'])
      };
      if (docCount > 0) {
        const percentiles = (0, _lodash.get)(aggregations, [...aggsPath, `${safeFieldName}_percentiles`, 'values'], []);
        if (percentiles && (0, _mlIsDefined.isDefined)(stats.min)) {
          const medianPercentile = (0, _lodash.find)(percentiles, {
            key: 50
          });
          stats.median = medianPercentile !== undefined ? medianPercentile.value : 0;
          stats.distribution = (0, _process_distribution_data.processDistributionData)(percentiles, _constants.PERCENTILE_SPACING, stats.min);
        }
      }
      batchStats.push(stats);
    });
    return batchStats;
  }));
};
exports.fetchNumericFieldsStats = fetchNumericFieldsStats;