"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.customPolicyResponseFailureEvaluator = void 0;
exports.isValidExampleOutput = isValidExampleOutput;
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function isValidExampleOutput(output) {
  // Check if output is an object and has the expected structure. It's defined in LangSmith, hence needs validation.
  return output && Array.isArray(output.insights) && output.insights.every(insight => {
    var _insight$remediation;
    return typeof insight === 'object' && insight !== null && 'group' in insight && 'events' in insight && Array.isArray(insight.events) && 'remediation' in insight && typeof ((_insight$remediation = insight.remediation) === null || _insight$remediation === void 0 ? void 0 : _insight$remediation.message) === 'string';
  });
}
function failWithComment(comment) {
  return {
    key: 'correct',
    score: 0,
    comment
  };
}
function tokenize(text) {
  return text.toLowerCase().replace(/[^\w\s]/g, ' ').split(/\s+/).filter(token => token.length > 2);
}
function cosineSimilarity(text1, text2) {
  const tokens1 = tokenize(text1);
  const tokens2 = tokenize(text2);
  if (tokens1.length === 0 || tokens2.length === 0) {
    return 0;
  }
  const allTokens = [...new Set([...tokens1, ...tokens2])];
  const vector1 = allTokens.map(token => tokens1.filter(t => t === token).length);
  const vector2 = allTokens.map(token => tokens2.filter(t => t === token).length);
  const dotProduct = vector1.reduce((sum, a, i) => sum + a * vector2[i], 0);
  const magnitude1 = Math.sqrt(vector1.reduce((sum, a) => sum + a * a, 0));
  const magnitude2 = Math.sqrt(vector2.reduce((sum, a) => sum + a * a, 0));
  return magnitude1 && magnitude2 ? dotProduct / (magnitude1 * magnitude2) : 0;
}
function getSimilarityScore(insight, requirement) {
  var _ref, _insight$remediation2, _ref2, _requirement$remediat;
  const message = (_ref = (_insight$remediation2 = insight.remediation) === null || _insight$remediation2 === void 0 ? void 0 : _insight$remediation2.message) !== null && _ref !== void 0 ? _ref : '';
  const reqMessage = (_ref2 = (_requirement$remediat = requirement.remediation) === null || _requirement$remediat === void 0 ? void 0 : _requirement$remediat.message) !== null && _ref2 !== void 0 ? _ref2 : '';
  if (!message || !reqMessage) {
    return 0;
  }
  return cosineSimilarity(message, reqMessage);
}
const customPolicyResponseFailureEvaluator = (run, example) => {
  var _run$outputs$insights, _run$outputs;
  const expectedOutput = example === null || example === void 0 ? void 0 : example.outputs;
  if (!isValidExampleOutput(expectedOutput)) {
    return failWithComment(_constants.EVALUATOR_ERRORS.INVALID_OUTPUT_STRUCTURE);
  }
  const {
    insights: requirements
  } = expectedOutput;
  const insights = ((_run$outputs$insights = (_run$outputs = run.outputs) === null || _run$outputs === void 0 ? void 0 : _run$outputs.insights) !== null && _run$outputs$insights !== void 0 ? _run$outputs$insights : []).reduce((acc, insight) => {
    acc[insight.group] = insight;
    return acc;
  }, {});
  if (Object.keys(insights).length === 0) {
    return failWithComment(_constants.EVALUATOR_ERRORS.NO_RESULTS);
  }
  const failedChecks = [];

  // check if insight count matches requirement count
  if (Object.keys(insights).length !== requirements.length) {
    failedChecks.push({
      label: 'number of insight groups does not match number of requirements',
      details: [`insights: ${Object.keys(insights).length}`, `requirements: ${requirements.length}`]
    });
  }
  const allSimilarityScores = [];
  for (const req of requirements) {
    const label = `requirement "${req.group}"`;
    const matchedInsight = insights[req.group];
    if (!matchedInsight) {
      failedChecks.push({
        label: `${label} did not match any insight group`
      });
    } else {
      var _ref3, _matchedInsight$remed, _req$remediation, _matchedInsight$event, _req$events;
      // check links
      const link = (_ref3 = matchedInsight === null || matchedInsight === void 0 ? void 0 : (_matchedInsight$remed = matchedInsight.remediation) === null || _matchedInsight$remed === void 0 ? void 0 : _matchedInsight$remed.link) !== null && _ref3 !== void 0 ? _ref3 : '';
      if (link !== ((_req$remediation = req.remediation) === null || _req$remediation === void 0 ? void 0 : _req$remediation.link)) {
        failedChecks.push({
          label: `Links for ${label} is not matching`,
          details: [`insight: ${JSON.stringify(matchedInsight)}`, `requirement: ${JSON.stringify(req)}`]
        });
      }

      // check events
      const events = (_matchedInsight$event = matchedInsight.events) !== null && _matchedInsight$event !== void 0 ? _matchedInsight$event : [];
      const reqEvents = (_req$events = req.events) !== null && _req$events !== void 0 ? _req$events : [];
      if (events.length !== reqEvents.length) {
        failedChecks.push({
          label: `Number of events for ${label} is not matching`,
          details: [`insight: ${JSON.stringify(matchedInsight)}`, `requirement: ${JSON.stringify(req)}`]
        });
      }
      for (const event of events) {
        const reqEvent = reqEvents.find(e => e.id === event.id);
        if (!reqEvent) {
          failedChecks.push({
            label: `Event with id "${event.id}" for ${label} is not matching`
          });
        }
        if ((reqEvent === null || reqEvent === void 0 ? void 0 : reqEvent.endpointId) !== event.endpointId) {
          failedChecks.push({
            label: `Event with id "${event.id}" for ${label} has different endpoint IDs`,
            details: [`insight: ${JSON.stringify(matchedInsight)}`, `requirement: ${JSON.stringify(req)}`]
          });
        }
        if ((reqEvent === null || reqEvent === void 0 ? void 0 : reqEvent.value) !== event.value) {
          failedChecks.push({
            label: `Event with id "${event.id}" for ${label} has different values`,
            details: [`insight: ${JSON.stringify(matchedInsight)}`, `requirement: ${JSON.stringify(req)}`]
          });
        }
      }
      const similarityScore = getSimilarityScore(matchedInsight, req);
      allSimilarityScores.push(similarityScore);
    }
  }
  const score = allSimilarityScores.length > 0 ? Number((allSimilarityScores.reduce((a, b) => a + b, 0) / allSimilarityScores.length).toFixed(2)) : 0;
  const comment = failedChecks.length ? `Failed checks:\n${failedChecks.map(c => {
    var _c$details;
    return (_c$details = c.details) !== null && _c$details !== void 0 && _c$details.length ? `${c.label}:\n  - ${c.details.join('\n  - ')}` : c.label;
  }).join('\n\n')}` : 'All checks passed';
  return {
    key: 'correct',
    score: failedChecks.length ? 0 : score,
    comment
  };
};
exports.customPolicyResponseFailureEvaluator = customPolicyResponseFailureEvaluator;