"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerReactEmbeddableFactory = exports.getReactEmbeddableFactory = void 0;
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const registry = {};

/**
 * Registers a new React embeddable factory. This should be called at plugin start time.
 *
 * @param type The key to register the factory under.
 * @param getFactory an async function that gets the factory definition for this key. This should always async import the
 * actual factory definition file to avoid polluting page load.
 */
const registerReactEmbeddableFactory = (type, getFactory) => {
  if (registry[type] !== undefined) throw new Error(_i18n.i18n.translate('embeddableApi.reactEmbeddable.factoryAlreadyExistsError', {
    defaultMessage: 'An embeddable factory for type: {key} is already registered.',
    values: {
      key: type
    }
  }));
  registry[type] = getFactory;
};
exports.registerReactEmbeddableFactory = registerReactEmbeddableFactory;
const getReactEmbeddableFactory = async key => {
  if (registry[key] === undefined) throw new Error(_i18n.i18n.translate('embeddableApi.reactEmbeddable.factoryNotFoundError', {
    defaultMessage: 'No embeddable factory found for type: {key}',
    values: {
      key
    }
  }));
  return registry[key]();
};
exports.getReactEmbeddableFactory = getReactEmbeddableFactory;