"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.workspaceSelector = exports.workspaceReducer = exports.workspaceInitializedSelector = exports.submitSearchSaga = exports.submitSearch = exports.initializeWorkspace = exports.fillWorkspaceSaga = void 0;
var _typescriptFsa = _interopRequireDefault(require("typescript-fsa"));
var _i18n = require("@kbn/i18n");
var _effects = require("redux-saga/effects");
var _typescriptFsaReducers = require("typescript-fsa-reducers");
var _reselect = require("reselect");
var _ = require(".");
var _global = require("./global");
var _datasource = require("./datasource");
var _fields = require("./fields");
var _fetch_top_nodes = require("../services/fetch_top_nodes");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const actionCreator = (0, _typescriptFsa.default)('x-pack/graph/workspace');
const initialWorkspaceState = {
  isInitialized: false
};
const initializeWorkspace = exports.initializeWorkspace = actionCreator('INITIALIZE_WORKSPACE');
const submitSearch = exports.submitSearch = actionCreator('SUBMIT_SEARCH');
const workspaceReducer = exports.workspaceReducer = (0, _typescriptFsaReducers.reducerWithInitialState)(initialWorkspaceState).case(_global.reset, () => ({
  isInitialized: false
})).case(initializeWorkspace, () => ({
  isInitialized: true
})).build();
const workspaceSelector = state => state.workspace;
exports.workspaceSelector = workspaceSelector;
const workspaceInitializedSelector = exports.workspaceInitializedSelector = (0, _reselect.createSelector)(workspaceSelector, workspace => workspace.isInitialized);

/**
 * Saga handling filling in top terms into workspace.
 *
 * It will load the top terms of the selected fields, add them to the workspace and fill in the connections.
 */
const fillWorkspaceSaga = ({
  getWorkspace,
  notifyReact,
  http,
  notifications
}) => {
  function* fetchNodes() {
    try {
      const workspace = getWorkspace();
      if (!workspace) {
        return;
      }
      const state = yield (0, _effects.select)();
      const fields = (0, _fields.selectedFieldsSelector)(state);
      const datasource = (0, _datasource.datasourceSelector)(state).current;
      if (datasource.type === 'none') {
        return;
      }
      const topTermNodes = yield (0, _effects.call)(_fetch_top_nodes.fetchTopNodes, http.post, datasource.title, fields);
      workspace.mergeGraph({
        nodes: topTermNodes,
        edges: []
      });
      yield (0, _effects.put)(initializeWorkspace());
      notifyReact();
      workspace.fillInGraph(fields.length * 10);
    } catch (e) {
      const message = 'body' in e ? e.body.message : e.message;
      notifications.toasts.addDanger({
        title: _i18n.i18n.translate('xpack.graph.fillWorkspaceError', {
          defaultMessage: 'Fetching top terms failed: {message}',
          values: {
            message
          }
        })
      });
    }
  }
  return function* () {
    yield (0, _effects.takeLatest)(_.fillWorkspace.match, fetchNodes);
  };
};
exports.fillWorkspaceSaga = fillWorkspaceSaga;
const submitSearchSaga = ({
  getWorkspace,
  handleSearchQueryError
}) => {
  function* submit(action) {
    const searchTerm = action.payload;
    yield (0, _effects.put)(initializeWorkspace());

    // type casting is safe, at this point workspace should be loaded
    const workspace = getWorkspace();
    const numHops = 2;
    const liveResponseFields = (0, _fields.liveResponseFieldsSelector)(yield (0, _effects.select)());
    if (searchTerm.startsWith('{')) {
      try {
        const query = JSON.parse(searchTerm);
        if (query.vertices) {
          // Is a graph explore request
          workspace.callElasticsearch(query);
        } else {
          // Is a regular query DSL query
          workspace.search(query, liveResponseFields, numHops);
        }
      } catch (err) {
        handleSearchQueryError(err);
      }
      return;
    }
    workspace.simpleSearch(searchTerm, liveResponseFields, numHops);
  }
  return function* () {
    yield (0, _effects.takeLatest)(submitSearch.match, submit);
  };
};
exports.submitSearchSaga = submitSearchSaga;