"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getUsefulGroups = getUsefulGroups;
var _constants = require("../constants");
var _get_review_fields = require("../review/get_review_fields");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Analyzes and processes an array of normalized columns to identify and retain
 * columns with meaningful data, collapsing the rest into a single GREEDYDATA column.
 */
function getUsefulGroups(columns) {
  let counter = 1;

  // Generates a unique ID for each field
  function uniqueId() {
    return String(counter++);
  }

  // Maps normalized columns to named columns by assigning unique IDs to tokens
  // and determining their patterns based on precedence.
  const namedColumns = columns.map(column => {
    return {
      tokens: column.tokens.map(({
        values,
        patterns
      }) => {
        const pattern = _constants.PATTERN_PRECEDENCE[patterns[0]];
        // If the pattern is a split character, return it as a literal value
        if (_constants.TOKEN_SPLIT_CHARS.includes(pattern)) {
          return {
            pattern
          };
        }
        // Otherwise, create a named field
        return {
          id: `field_${uniqueId()}`,
          component: pattern,
          values
        };
      }),
      whitespace: column.whitespace
    };
  });

  // Identifies the subset of columns that are "useful" based on two criteria:
  // 1. Columns surrounded by variable whitespace (indicating intentional separation).
  // 2. Columns containing tokens with patterns that are not collapsible.
  const usefulColumns = namedColumns.slice(0, Math.max(namedColumns.findLastIndex(col => {
    const leadingWhitespaceRange = col.whitespace.maxLeading - col.whitespace.minLeading;
    const trailingWhitespaceRange = col.whitespace.maxTrailing - col.whitespace.minTrailing;
    return leadingWhitespaceRange > 0 || trailingWhitespaceRange > 0;
  }), namedColumns.findLastIndex(column => {
    return column.tokens.some(node => {
      if ((0, _utils.isNamedField)(node)) {
        return !(0, _get_review_fields.isCollapsiblePattern)(node.component);
      }
      return node.pattern.trim();
    });
  }) + 1));

  // If there are remaining columns after the "useful" ones, collapse them into
  // a single GREEDYDATA column and append it to the useful columns.
  if (usefulColumns.length < namedColumns.length) {
    const greedyDataNamedField = namedColumns.slice(usefulColumns.length).flatMap(col => col.tokens).find(_utils.isNamedField);
    usefulColumns.push({
      tokens: [{
        id: greedyDataNamedField ? greedyDataNamedField.id : `field_${uniqueId()}`,
        component: 'GREEDYDATA',
        values: []
      }],
      whitespace: {
        minLeading: namedColumns[usefulColumns.length].whitespace.minLeading,
        maxLeading: namedColumns[usefulColumns.length].whitespace.maxLeading,
        minTrailing: namedColumns[namedColumns.length - 1].whitespace.minTrailing,
        maxTrailing: namedColumns[namedColumns.length - 1].whitespace.maxTrailing
      }
    });
  }
  return usefulColumns;
}