"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertAuthorToRole = convertAuthorToRole;
exports.convertBaseMessagesToContent = convertBaseMessagesToContent;
exports.convertMessageContentToParts = convertMessageContentToParts;
exports.convertResponseBadFinishReasonToErrorMsg = convertResponseBadFinishReasonToErrorMsg;
exports.convertResponseContentToChatGenerationChunk = convertResponseContentToChatGenerationChunk;
exports.getMessageAuthor = getMessageAuthor;
exports.parseGeminiStreamAsAsyncIterator = exports.parseGeminiStream = exports.parseGeminiResponse = void 0;
var _generativeAi = require("@google/generative-ai");
var _messages = require("@langchain/core/messages");
var _outputs = require("@langchain/core/outputs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function convertResponseContentToChatGenerationChunk(response, extra) {
  var _content$parts$0$text, _content$parts$;
  if (!response.candidates || response.candidates.length === 0) {
    return null;
  }
  const functionCalls = response.functionCalls();
  const [candidate] = response.candidates;
  const {
    content,
    ...generationInfo
  } = candidate;
  const text = (_content$parts$0$text = content === null || content === void 0 ? void 0 : (_content$parts$ = content.parts[0]) === null || _content$parts$ === void 0 ? void 0 : _content$parts$.text) !== null && _content$parts$0$text !== void 0 ? _content$parts$0$text : '';
  const toolCallChunks = [];
  if (functionCalls) {
    toolCallChunks.push(...functionCalls.map(fc => ({
      ...fc,
      args: JSON.stringify(fc.args),
      index: extra.index,
      type: 'tool_call_chunk'
    })));
  }
  return new _outputs.ChatGenerationChunk({
    text,
    message: new _messages.AIMessageChunk({
      content: text,
      name: !content ? undefined : content.role,
      tool_call_chunks: toolCallChunks,
      // Each chunk can have unique "generationInfo", and merging strategy is unclear,
      // so leave blank for now.
      additional_kwargs: {},
      usage_metadata: extra.usageMetadata
    }),
    generationInfo
  });
}
function convertAuthorToRole(author) {
  switch (author) {
    /**
     *  Note: Gemini currently is not supporting system messages
     *  we will convert them to human messages and merge with following
     * */
    case 'ai':
    case 'model':
      // getMessageAuthor returns message.name. code ex.: return message.name ?? type;
      return 'model';
    case 'system':
    case 'human':
      return 'user';
    case 'tool':
    case 'function':
      return 'function';
    default:
      throw new Error(`Unknown / unsupported author: ${author}`);
  }
}
function convertBaseMessagesToContent(messages, isMultimodalModel) {
  return messages.reduce((acc, message, index) => {
    if (!(0, _messages.isBaseMessage)(message)) {
      throw new Error('Unsupported message input');
    }
    const author = getMessageAuthor(message);
    if (author === 'system' && index !== 0) {
      throw new Error('System message should be the first one');
    }
    const role = convertAuthorToRole(author);
    const parts = convertMessageContentToParts(message, isMultimodalModel);
    if (acc.mergeWithPreviousContent) {
      const prevContent = acc.content[acc.content.length - 1];
      if (!prevContent) {
        throw new Error('There was a problem parsing your system message. Please try a prompt without one.');
      }
      prevContent.parts.push(...parts);
      return {
        mergeWithPreviousContent: false,
        content: acc.content
      };
    }
    let actualRole = role;
    if (actualRole === 'function') {
      // GenerativeAI API will throw an error if the role is not "user" or "model."
      actualRole = 'user';
    }
    const content = {
      role: actualRole,
      parts
    };
    return {
      mergeWithPreviousContent: author === 'system',
      content: [...acc.content, content]
    };
  }, {
    content: [],
    mergeWithPreviousContent: false
  }).content;
}
function convertMessageContentToParts(message, isMultimodalModel) {
  if (typeof message.content === 'string' && message.content !== '') {
    return [{
      text: message.content
    }];
  }
  let functionCalls = [];
  let functionResponses = [];
  let messageParts = [];
  if ('tool_calls' in message && Array.isArray(message.tool_calls) && message.tool_calls.length > 0) {
    functionCalls = message.tool_calls.map(tc => ({
      functionCall: {
        name: tc.name,
        args: tc.args
      }
    }));
  } else if (message._getType() === 'tool' && message.name && message.content) {
    functionResponses = [{
      functionResponse: {
        name: message.name,
        response: message.content
      }
    }];
  } else if (Array.isArray(message.content)) {
    messageParts = message.content.map(c => {
      if (c.type === 'text') {
        return {
          text: c.text
        };
      }
      if (c.type === 'image_url') {
        if (!isMultimodalModel) {
          throw new Error(`This model does not support images`);
        }
        let source;
        if (typeof c.image_url === 'string') {
          source = c.image_url;
        } else if (typeof c.image_url === 'object' && 'url' in c.image_url) {
          source = c.image_url.url;
        } else {
          throw new Error('Please provide image as base64 encoded data URL');
        }
        const [dm, data] = source.split(',');
        if (!dm.startsWith('data:')) {
          throw new Error('Please provide image as base64 encoded data URL');
        }
        const [mimeType, encoding] = dm.replace(/^data:/, '').split(';');
        if (encoding !== 'base64') {
          throw new Error('Please provide image as base64 encoded data URL');
        }
        return {
          inlineData: {
            data,
            mimeType
          }
        };
      } else if (c.type === 'media') {
        return messageContentMedia(c);
      } else if (c.type === 'tool_use') {
        return {
          functionCall: {
            name: c.name,
            args: c.input
          }
        };
      }
      throw new Error(`Unknown content type ${c.type}`);
    });
  }
  return [...messageParts, ...functionCalls, ...functionResponses];
}
function getMessageAuthor(message) {
  var _message$name;
  const type = message._getType();
  if (_messages.ChatMessage.isInstance(message)) {
    return message.role;
  }
  if (type === 'tool') {
    return type;
  }
  return (_message$name = message.name) !== null && _message$name !== void 0 ? _message$name : type;
}

// will be removed once FileDataPart is supported in @langchain/google-genai
function messageContentMedia(content) {
  if ('mimeType' in content && 'data' in content) {
    return {
      inlineData: {
        mimeType: content.mimeType,
        data: content.data
      }
    };
  }
  throw new Error('Invalid media content');
}

// TODO Google's TS library is behind the API
// remove this enum once the library is updated
// https://github.com/google-gemini/generative-ai-js/pull/270
var FinishReasonMore = /*#__PURE__*/function (FinishReasonMore) {
  FinishReasonMore["BLOCKLIST"] = "BLOCKLIST";
  FinishReasonMore["PROHIBITED_CONTENT"] = "PROHIBITED_CONTENT";
  FinishReasonMore["SPII"] = "SPII";
  FinishReasonMore["MALFORMED_FUNCTION_CALL"] = "MALFORMED_FUNCTION_CALL";
  return FinishReasonMore;
}(FinishReasonMore || {});
const badFinishReasons = [_generativeAi.FinishReason.RECITATION, _generativeAi.FinishReason.SAFETY, FinishReasonMore.BLOCKLIST, FinishReasonMore.PROHIBITED_CONTENT, FinishReasonMore.SPII, FinishReasonMore.MALFORMED_FUNCTION_CALL];
function hadBadFinishReason(candidate) {
  return !!candidate.finishReason && badFinishReasons.includes(candidate.finishReason);
}
function convertResponseBadFinishReasonToErrorMsg(response) {
  if (response.candidates && response.candidates.length > 0) {
    const candidate = response.candidates[0];
    if (hadBadFinishReason(candidate)) {
      var _candidate$safetyRati, _candidate$safetyRati2;
      if (candidate.finishReason === _generativeAi.FinishReason.SAFETY && candidate.safetyRatings && ((_candidate$safetyRati = (_candidate$safetyRati2 = candidate.safetyRatings) === null || _candidate$safetyRati2 === void 0 ? void 0 : _candidate$safetyRati2.length) !== null && _candidate$safetyRati !== void 0 ? _candidate$safetyRati : 0) > 0) {
        const safetyReasons = getSafetyReasons(candidate.safetyRatings);
        return `Gemini Utils: action result status is error. Candidate was blocked due to ${candidate.finishReason} - ${safetyReasons}`;
      } else {
        return `Gemini Utils: action result status is error. Candidate was blocked due to ${candidate.finishReason}`;
      }
    }
  }
  return null;
}

// not sure why these properties are not on the type, as they are on the data

const getSafetyReasons = safetyRatings => {
  const reasons = safetyRatings.filter(t => t.blocked);
  return reasons.reduce((acc, t, i) => `${acc.length ? `${acc} ` : ''}${t.category}: ${t.severity}${i < reasons.length - 1 ? ',' : ''}`, '');
};
const parseGeminiStreamAsAsyncIterator = async function* (stream, logger, abortSignal) {
  if (abortSignal) {
    abortSignal.addEventListener('abort', () => {
      stream.destroy();
    });
  }
  try {
    for await (const chunk of stream) {
      const decoded = chunk.toString();
      const parsed = parseGeminiResponse(decoded);
      // Split the parsed string into chunks of 5 characters
      for (let i = 0; i < parsed.length; i += 5) {
        yield parsed.substring(i, i + 5);
      }
    }
  } catch (err) {
    if (abortSignal !== null && abortSignal !== void 0 && abortSignal.aborted) {
      logger.info('Gemini stream parsing was aborted.');
    } else {
      throw err;
    }
  }
};
exports.parseGeminiStreamAsAsyncIterator = parseGeminiStreamAsAsyncIterator;
const parseGeminiStream = async (stream, logger, abortSignal, tokenHandler) => {
  let responseBody = '';
  stream.on('data', chunk => {
    const decoded = chunk.toString();
    const parsed = parseGeminiResponse(decoded);
    if (tokenHandler) {
      // Split the parsed string into chunks of 5 characters
      for (let i = 0; i < parsed.length; i += 5) {
        tokenHandler(parsed.substring(i, i + 5));
      }
    }
    responseBody += parsed;
  });
  return new Promise((resolve, reject) => {
    stream.on('end', () => {
      resolve(responseBody);
    });
    stream.on('error', err => {
      reject(err);
    });
    if (abortSignal) {
      abortSignal.addEventListener('abort', () => {
        logger.info('Bedrock stream parsing was aborted.');
        stream.destroy();
        resolve(responseBody);
      });
    }
  });
};

/** Parse Gemini stream response body */
exports.parseGeminiStream = parseGeminiStream;
const parseGeminiResponse = responseBody => {
  return responseBody.split('\n').filter(line => line.startsWith('data: ') && !line.endsWith('[DONE]')).map(line => JSON.parse(line.replace('data: ', ''))).filter(line => 'candidates' in line).reduce((prev, line) => {
    if (line.candidates[0] && line.candidates[0].content) {
      const parts = line.candidates[0].content.parts;
      const text = parts.map(part => part.text).join('');
      return prev + text;
    }
    return prev;
  }, '');
};
exports.parseGeminiResponse = parseGeminiResponse;