"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeEditApi = initializeEditApi;
var _presentationPublishing = require("@kbn/presentation-publishing");
var _esqlUtils = require("@kbn/esql-utils");
var _lodash = require("lodash");
var _public = require("@kbn/embeddable-plugin/public");
var _i18n = require("@kbn/i18n");
var _rxjs = require("rxjs");
var _constants = require("../../../common/constants");
var _helper = require("../helper");
var _setup_inline_editing = require("../inline_editing/setup_inline_editing");
var _panel_management = require("../inline_editing/panel_management");
var _mount = require("../mount");
var _type_guards = require("../type_guards");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getSupportedTriggers(getState, visualizationMap) {
  return () => {
    var _currentState$attribu;
    const currentState = getState();
    if ((_currentState$attribu = currentState.attributes) !== null && _currentState$attribu !== void 0 && _currentState$attribu.visualizationType) {
      var _visualizationMap$cur;
      return ((_visualizationMap$cur = visualizationMap[currentState.attributes.visualizationType]) === null || _visualizationMap$cur === void 0 ? void 0 : _visualizationMap$cur.triggers) || [];
    }
    return [];
  };
}
function isReadOnly(viewMode$) {
  return viewMode$.getValue() === 'view';
}
function isEditMode(viewMode$) {
  return viewMode$.getValue() === 'edit';
}
function hasManagedApi(api) {
  return Boolean(api && typeof api.isManaged === 'boolean');
}

/**
 * Initialize the edit API for the embeddable
 **/
function initializeEditApi(uuid, initialState, getState, internalApi, stateApi, inspectorApi, searchContextApi, isTextBasedLanguage, startDependencies, parentApi) {
  const supportedTriggers = getSupportedTriggers(getState, startDependencies.visualizationMap);
  const isManaged = currentState => {
    return currentState.managed || (hasManagedApi(parentApi) ? parentApi.isManaged : false);
  };
  const isESQLModeEnabled = () => uiSettings.get(_esqlUtils.ENABLE_ESQL);
  const viewMode$ = (0, _helper.extractInheritedViewModeObservable)(parentApi);
  const {
    disabledActionIds$,
    setDisabledActionIds
  } = (0, _presentationPublishing.apiPublishesDisabledActionIds)(parentApi) ? parentApi : {
    disabledActionIds$: new _rxjs.BehaviorSubject(undefined),
    setDisabledActionIds: _lodash.noop
  };
  if (isTextBasedLanguage(initialState)) {
    var _disabledActionIds$$g;
    // do not expose the drilldown action for ES|QL
    setDisabledActionIds(disabledActionIds$ === null || disabledActionIds$ === void 0 ? void 0 : (_disabledActionIds$$g = disabledActionIds$.getValue()) === null || _disabledActionIds$$g === void 0 ? void 0 : _disabledActionIds$$g.concat(['OPEN_FLYOUT_ADD_DRILLDOWN']));
  }

  /**
   * Inline editing section
   */
  const navigateToLensEditor = (stateTransfer, skipAppLeave) => async () => {
    var _parentApiContext$cur, _parentApiContext$get;
    if (!parentApi || !(0, _presentationPublishing.apiHasAppContext)(parentApi)) {
      return;
    }
    const parentApiContext = parentApi.getAppContext();
    const currentState = getState();
    await stateTransfer.navigateToEditor(_constants.APP_ID, {
      path: (0, _constants.getEditPath)(currentState.savedObjectId),
      state: {
        embeddableId: uuid,
        valueInput: currentState,
        originatingApp: (_parentApiContext$cur = parentApiContext.currentAppId) !== null && _parentApiContext$cur !== void 0 ? _parentApiContext$cur : 'dashboards',
        originatingPath: (_parentApiContext$get = parentApiContext.getCurrentPath) === null || _parentApiContext$get === void 0 ? void 0 : _parentApiContext$get.call(parentApiContext),
        searchSessionId: currentState.searchSessionId
      },
      skipAppLeave
    });
  };
  const panelManagementApi = (0, _panel_management.setupPanelManagement)(uuid, parentApi, {
    isNewlyCreated$: internalApi.isNewlyCreated$,
    setAsCreated: internalApi.setAsCreated,
    isReadOnly: () => isReadOnly(viewMode$),
    canEdit: () => isEditMode(viewMode$)
  });
  const updateState = newState => {
    stateApi.updateAttributes(newState.attributes);
    stateApi.updateSavedObjectId(newState.savedObjectId);
  };

  /**
   * Use the search context api here for filters for 2 reasons:
   *  - the filters here have the correct references already injected
   *  - the edit filters flow may change in the future and this is the right place to get the filters
   */
  const getFilters = ({
    attributes
  }) => {
    var _searchContextApi$fil;
    return (_searchContextApi$fil = searchContextApi.filters$.getValue()) !== null && _searchContextApi$fil !== void 0 ? _searchContextApi$fil : attributes.state.filters;
  };
  const convertVisualizationState = ({
    attributes
  }) => {
    var _startDependencies$vi, _attributes$visualiza;
    const visState = attributes.state.visualization;
    const convertToRuntimeState = (_startDependencies$vi = startDependencies.visualizationMap[(_attributes$visualiza = attributes.visualizationType) !== null && _attributes$visualiza !== void 0 ? _attributes$visualiza : '']) === null || _startDependencies$vi === void 0 ? void 0 : _startDependencies$vi.convertToRuntimeState;
    if (!convertToRuntimeState) return visState;
    return convertToRuntimeState(visState, attributes.state.datasourceStates);
  };

  /**
   * Wrap getState() when inline editing to ensure:
   *  - Filters in the attributes are properly injected with the correct references to avoid
   *    issues when saving/navigating to the full editor
   *  - Apply runtime conversions to visualization state
   */
  const getModifiedState = () => {
    const currentState = getState();
    return {
      ...currentState,
      attributes: {
        ...currentState.attributes,
        state: {
          ...currentState.attributes.state,
          filters: getFilters(currentState),
          visualization: convertVisualizationState(currentState)
        }
      }
    };
  };

  // This will handle both edit and read only mode based on the view mode
  const getInlineEditor = (0, _setup_inline_editing.prepareInlineEditPanel)(initialState, getModifiedState, updateState, internalApi, panelManagementApi, inspectorApi, startDependencies, navigateToLensEditor, uuid, parentApi);

  /**
   * The rest of the edit stuff
   */
  const {
    uiSettings,
    capabilities,
    data
  } = startDependencies;
  const canEdit = () => {
    var _capabilities$dashboa;
    if (!isEditMode(viewMode$)) {
      return false;
    }
    const currentState = getState();
    // check if it's in ES|QL mode
    if (isTextBasedLanguage(currentState) && !isESQLModeEnabled()) {
      return false;
    }
    if (isManaged(currentState)) {
      return false;
    }
    return Boolean(capabilities.visualize_v2.save) || !getState().savedObjectId && Boolean((_capabilities$dashboa = capabilities.dashboard_v2) === null || _capabilities$dashboa === void 0 ? void 0 : _capabilities$dashboa.showWriteControls) && Boolean(capabilities.visualize_v2.show);
  };
  const canShowConfig = () => {
    return isReadOnly(viewMode$) && Boolean(capabilities.visualize_v2.show);
  };
  const getEditPanel = async ({
    showOnly,
    closeFlyout
  } = {
    closeFlyout: _lodash.noop
  }) => {
    // save the initial state in case it needs to revert later on
    const firstState = getState();
    const ConfigPanel = await getInlineEditor({
      // restore the first state found when the panel opened
      onCancel: () => updateState({
        ...firstState
      }),
      // the getState() here contains the wrong filters references but the input attributes
      // are correct as getInlineEditor() handler is using the getModifiedState() function
      onApply: showOnly ? _lodash.noop : attributes => updateState({
        ...getState(),
        attributes
      }),
      closeFlyout
    });
    return ConfigPanel !== null && ConfigPanel !== void 0 ? ConfigPanel : undefined;
  };
  return {
    api: {
      uuid,
      viewMode$,
      getTypeDisplayName: () => _i18n.i18n.translate('xpack.lens.embeddableDisplayName', {
        defaultMessage: 'visualization'
      }),
      supportedTriggers,
      disabledActionIds$,
      setDisabledActionIds,
      /**
       * This is the key method to enable the new Editing capabilities API
       * Lens will leverage the neutral nature of this function to build the inline editing experience
       */
      onEdit: async () => {
        if (!parentApi || !(0, _presentationPublishing.apiHasAppContext)(parentApi)) {
          return;
        }

        // this will force the embeddable to toggle the inline editing feature
        const canEditInline = (0, _type_guards.apiPublishesInlineEditingCapabilities)(parentApi) ? parentApi.canEditInline : true;

        // just navigate directly to the editor
        if (!canEditInline) {
          const navigateFn = navigateToLensEditor(new _public.EmbeddableStateTransfer(startDependencies.coreStart.application.navigateToApp, startDependencies.coreStart.application.currentAppId$), true);
          return navigateFn();
        }
        (0, _mount.mountInlinePanel)({
          core: startDependencies.coreStart,
          api: parentApi,
          loadContent: getEditPanel,
          options: {
            uuid
          }
        });
      },
      getEditPanel,
      /**
       * Check everything here: user/app permissions and the current inline editing state
       */
      isEditingEnabled: () => {
        return Boolean(parentApi && (0, _presentationPublishing.apiHasAppContext)(parentApi) && canEdit() && panelManagementApi.isEditingEnabled());
      },
      isReadOnlyEnabled: () => {
        var _capabilities$dashboa2;
        return {
          read: Boolean(parentApi && (0, _presentationPublishing.apiHasAppContext)(parentApi) && canShowConfig()),
          write: Boolean(((_capabilities$dashboa2 = capabilities.dashboard_v2) === null || _capabilities$dashboa2 === void 0 ? void 0 : _capabilities$dashboa2.showWriteControls) && !isManaged(getState()))
        };
      },
      onShowConfig: async () => {
        if (!parentApi || !(0, _presentationPublishing.apiHasAppContext)(parentApi)) {
          return;
        }
        (0, _mount.mountInlinePanel)({
          core: startDependencies.coreStart,
          api: parentApi,
          loadContent: async ({
            closeFlyout
          } = {
            closeFlyout: _lodash.noop
          }) => {
            return getEditPanel({
              showOnly: true,
              closeFlyout
            });
          },
          options: {
            uuid
          }
        });
      },
      getEditHref: async () => {
        if (!parentApi || !(0, _presentationPublishing.apiHasAppContext)(parentApi)) {
          return;
        }
        const currentState = getState();
        return (0, _constants.getEditPath)(currentState.savedObjectId, currentState.timeRange, currentState.filters, data.query.timefilter.timefilter.getRefreshInterval());
      }
    }
  };
}