"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDataBounds = getDataBounds;
exports.hasNumericHistogramDimension = hasNumericHistogramDimension;
exports.validateAxisDomain = validateAxisDomain;
exports.validateExtent = validateExtent;
exports.validateLogarithmicExtent = validateLogarithmicExtent;
exports.validateZeroInclusivityExtent = validateZeroInclusivityExtent;
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Returns true if the provided extent includes 0
 * @param extent
 * @returns boolean
 */
function validateZeroInclusivityExtent(extent) {
  return extent && extent.lowerBound != null && extent.upperBound != null && extent.lowerBound <= 0 && extent.upperBound >= 0;
}

/**
 * Returns true if the provided extent includes 0
 * @param extent
 * @returns boolean
 */
function validateLogarithmicExtent(extent) {
  return extent && extent.lowerBound != null && extent.upperBound != null && (extent.lowerBound < 0 && extent.upperBound < 0 || extent.lowerBound > 0 && extent.upperBound > 0);
}

/**
 * Returns true if the provided extent is a valid range
 * @param extent
 * @returns boolean
 */
function validateAxisDomain(extents) {
  return extents && extents.lowerBound != null && extents.upperBound != null && extents.upperBound > extents.lowerBound;
}
/**
 * Returns true if the provided column is a numeric histogram dimension
 * @param extent
 * @returns boolean
 */
function hasNumericHistogramDimension(datasourceLayer, columnId) {
  if (!columnId) {
    return false;
  }
  const operation = datasourceLayer === null || datasourceLayer === void 0 ? void 0 : datasourceLayer.getOperationForColumnId(columnId);
  return Boolean(operation && operation.dataType === 'number' && operation.scale === 'interval');
}

/**
 * Finds the table data min and max. Returns undefined when no min/max is found
 * @param layerId
 * @param tables
 * @param columnId
 * @returns
 */
function getDataBounds(layerId, tables, columnId) {
  const table = tables === null || tables === void 0 ? void 0 : tables[layerId];
  if (columnId && table) {
    const sortedRows = table.rows.map(({
      [columnId]: value
    }) => value)
    // ignore missing hit
    .filter(v => v != null).sort((a, b) => a - b);
    if (sortedRows.length) {
      return {
        min: sortedRows[0],
        max: sortedRows[sortedRows.length - 1]
      };
    }
  }
}
function validateExtent(hasBarOrArea, extent, scaleType) {
  if (!validateAxisDomain(extent)) {
    return {
      errorMsg: _i18n.i18n.translate('xpack.lens.axisExtent.boundaryError', {
        defaultMessage: 'Lower bound should be less than upper bound'
      })
    };
  }
  if (scaleType === 'log') {
    const key = validateLogarithmicExtent(extent) ? 'helpMsg' : 'errorMsg';
    return {
      [key]: _i18n.i18n.translate('xpack.lens.axisExtent.logExcludeZero', {
        defaultMessage: 'Logarithmic bounds must not contain zero'
      })
    };
  }
  if (hasBarOrArea) {
    const key = validateZeroInclusivityExtent(extent) ? 'helpMsg' : 'errorMsg';
    return {
      [key]: _i18n.i18n.translate('xpack.lens.axisExtent.inclusiveZero', {
        defaultMessage: 'Bounds must contain zero'
      })
    };
  }
  return {};
}