"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerLensVisualizationsDeleteAPIRoute = void 0;
var _boom = require("@hapi/boom");
var _constants = require("../../../../common/constants");
var _schema = require("./schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerLensVisualizationsDeleteAPIRoute = (router, {
  contentManagement
}) => {
  const deleteRoute = router.delete({
    path: `${_constants.LENS_VIS_API_PATH}/{id}`,
    access: _constants.LENS_API_ACCESS,
    enableQueryVersion: true,
    summary: 'Delete Lens visualization',
    description: 'Delete a Lens visualization by id.',
    options: {
      tags: ['oas-tag:Lens'],
      availability: {
        stability: 'experimental'
      }
    },
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on Content Client for authorization'
      }
    }
  });
  deleteRoute.addVersion({
    version: _constants.LENS_API_VERSION,
    validate: {
      request: {
        params: _schema.lensDeleteRequestParamsSchema
      },
      response: {
        204: {
          description: 'No Content'
        },
        400: {
          description: 'Malformed request'
        },
        401: {
          description: 'Unauthorized'
        },
        403: {
          description: 'Forbidden'
        },
        404: {
          description: 'Resource not found'
        },
        500: {
          description: 'Internal Server Error'
        }
      }
    }
  }, async (ctx, req, res) => {
    // TODO fix IContentClient to type this client based on the actual
    const client = contentManagement.contentClient.getForRequest({
      request: req,
      requestHandlerContext: ctx
    }).for(_constants.LENS_CONTENT_TYPE);
    try {
      const {
        result
      } = await client.delete(req.params.id);
      if (!result.success) {
        throw new Error(`Failed to delete Lens visualization with id [${req.params.id}].`);
      }
      return res.noContent();
    } catch (error) {
      if ((0, _boom.isBoom)(error)) {
        if (error.output.statusCode === 404) {
          return res.notFound({
            body: {
              message: `A Lens visualization with id [${req.params.id}] was not found.`
            }
          });
        }
        if (error.output.statusCode === 403) {
          return res.forbidden();
        }
      }
      return (0, _boom.boomify)(error); // forward unknown error
    }
  });
};
exports.registerLensVisualizationsDeleteAPIRoute = registerLensVisualizationsDeleteAPIRoute;