"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.QuickLensJobCreator = void 0;
var _i18n = require("@kbn/i18n");
var _esQuery = require("@kbn/es-query");
var _default_configs = require("../common/job_creator/util/default_configs");
var _new_job = require("../../../../../common/constants/new_job");
var _utils = require("./utils");
var _visualization_extractor = require("./visualization_extractor");
var _job_from_dashboard = require("../job_from_dashboard");
var _job_cloning_service = require("../../../services/job_cloning_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class QuickLensJobCreator extends _job_from_dashboard.QuickJobCreatorBase {
  constructor(lens, dataViews, kibanaConfig, timeFilter, share, mlApi) {
    super(dataViews, kibanaConfig, timeFilter, share, mlApi);
    this.lens = lens;
  }
  async createAndSaveJob(jobId, bucketSpan, embeddable, startJob, runInRealTime, layerIndex) {
    const {
      query,
      filters,
      to,
      from,
      dashboard,
      chartInfo
    } = await (0, _utils.getJobsItemsFromEmbeddable)(embeddable, this.lens);
    if (query === undefined || filters === undefined) {
      throw new Error('Cannot create job, query and filters are undefined');
    }
    const {
      jobConfig,
      datafeedConfig,
      start,
      end,
      jobType
    } = await this.createJob(chartInfo, from, to, query, filters, bucketSpan, layerIndex);
    const createdByLabel = jobType === _new_job.JOB_TYPE.SINGLE_METRIC ? _new_job.CREATED_BY_LABEL.SINGLE_METRIC_FROM_LENS : _new_job.CREATED_BY_LABEL.MULTI_METRIC_FROM_LENS;
    const result = await this.putJobAndDataFeed({
      jobId,
      datafeedConfig,
      jobConfig,
      createdByLabel,
      dashboard,
      start,
      end,
      startJob,
      runInRealTime
    });
    return result;
  }
  async createAndStashADJob(vis, startString, endString, query, filters, layerIndex) {
    const chartInfo = await (0, _utils.getChartInfoFromVisualization)(this.lens, vis);
    try {
      const {
        jobConfig,
        datafeedConfig,
        jobType,
        start,
        end,
        includeTimeRange
      } = await this.createJob(chartInfo, startString, endString, query, filters, _new_job.DEFAULT_BUCKET_SPAN, layerIndex);

      // add job config and start and end dates to the
      // job cloning stash, so they can be used
      // by the new job wizards
      _job_cloning_service.jobCloningService.stashJobForCloning({
        jobConfig,
        datafeedConfig,
        createdBy: jobType === _new_job.JOB_TYPE.SINGLE_METRIC ? _new_job.CREATED_BY_LABEL.SINGLE_METRIC : _new_job.CREATED_BY_LABEL.MULTI_METRIC,
        start,
        end
      }, true, includeTimeRange, !includeTimeRange);
    } catch (error) {
      // eslint-disable-next-line no-console
      console.error(error);
    }
  }
  async createJob(chartInfo, startString, endString, query, filters, bucketSpan, layerIndex) {
    const {
      jobConfig,
      datafeedConfig,
      jobType
    } = await this.createADJobFromLensSavedObject(chartInfo, query, filters, bucketSpan, layerIndex);
    let start;
    let end;
    let includeTimeRange = true;
    try {
      // attempt to parse the start and end dates.
      // if start and end values cannot be determined
      // instruct the job cloning code to auto-select the
      // full time range for the index.
      const {
        min,
        max
      } = this.timeFilter.calculateBounds({
        to: endString,
        from: startString
      });
      start = min === null || min === void 0 ? void 0 : min.valueOf();
      end = max === null || max === void 0 ? void 0 : max.valueOf();
      if (start === undefined || end === undefined || isNaN(start) || isNaN(end)) {
        throw Error(_i18n.i18n.translate('xpack.ml.newJob.fromLens.createJob.error.timeRange', {
          defaultMessage: 'Incompatible time range'
        }));
      }
    } catch (error) {
      // eslint-disable-next-line no-console
      console.error(error);
      includeTimeRange = false;
      start = undefined;
      end = undefined;
    }
    return {
      jobConfig,
      datafeedConfig,
      jobType,
      start,
      end,
      includeTimeRange
    };
  }
  async createADJobFromLensSavedObject(chartInfo, query, filters, bucketSpan, layerIndex) {
    var _timeField$operation$;
    // we should not have got this far if the query is ES|QL
    // but just in case, throw an error if it is
    if ((0, _esQuery.isOfAggregateQueryType)(chartInfo.query)) {
      throw new Error('Cannot create job, query is of aggregate type');
    }
    const compatibleLayers = chartInfo.layers.filter(_utils.isCompatibleLayer);
    const selectedLayer = layerIndex !== undefined ? chartInfo.layers[layerIndex] : compatibleLayers[0];
    const visExtractor = new _visualization_extractor.VisualizationExtractor();
    const {
      fields,
      timeField,
      splitField,
      dataView
    } = await visExtractor.extractFields(selectedLayer);
    const jobConfig = (0, _default_configs.createEmptyJob)();
    const datafeedConfig = (0, _default_configs.createEmptyDatafeed)(dataView.getIndexPattern());
    const combinedFiltersAndQueries = this.combineQueriesAndFilters({
      query,
      filters
    }, {
      query: chartInfo.query,
      filters: chartInfo.filters
    }, dataView);
    datafeedConfig.query = combinedFiltersAndQueries;
    jobConfig.analysis_config.detectors = (0, _utils.createDetectors)(fields, splitField);
    jobConfig.data_description.time_field = (_timeField$operation$ = timeField.operation.fields) === null || _timeField$operation$ === void 0 ? void 0 : _timeField$operation$[0];
    jobConfig.analysis_config.bucket_span = bucketSpan;
    if (splitField && splitField.operation.fields) {
      jobConfig.analysis_config.influencers = [splitField.operation.fields[0]];
    }
    const isSingleMetric = !splitField && jobConfig.analysis_config.detectors.length === 1;
    const jobType = isSingleMetric ? _new_job.JOB_TYPE.SINGLE_METRIC : _new_job.JOB_TYPE.MULTI_METRIC;
    if (isSingleMetric) {
      jobConfig.model_plot_config = {
        enabled: true,
        annotations_enabled: true
      };
    }
    return {
      jobConfig,
      datafeedConfig,
      jobType
    };
  }
}
exports.QuickLensJobCreator = QuickLensJobCreator;