"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LogstashMetricbeatMonitoring = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _create_query = require("./create_query");
var _get_high_level_stats = require("./get_high_level_stats");
var _constants = require("../../common/constants");
var _logstash_monitoring = require("./logstash_monitoring");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class LogstashMetricbeatMonitoring {
  constructor() {
    (0, _defineProperty2.default)(this, "indexPattern", {
      state: _constants.INDEX_PATTERN_LOGSTASH_STACK_MONITORING_STATE,
      stats: _constants.INDEX_PATTERN_LOGSTASH_STACK_MONITORING_STATS
    });
    (0, _defineProperty2.default)(this, "monitoringType", 'metricbeat');
  }
  /*
   * Call the function for fetching and summarizing Logstash metrics for Metricbeat monitoring
   * @param {Object} callCluster - ES client
   * @param {Array} clusterUuids - List cluster UUIDs to retrieve metrics
   * @param {string} monitoringClusterUuid - monitoring cluster UUID
   * @param {string} start - start timestamp
   * @param {string} end - end timestamp
   * @param {Object} options - additional processing required options
   * @return {Object} - Logstash stats in an object keyed by the cluster UUIDs
   * Note that, we try to fetch all metrics for the given time regardless of the cluster UUID
   * If metrics do not have UUID, metrics will be included in the monitoring cluster UUID
   */
  async collectMetrics(callCluster, clusterUuids, monitoringClusterUuid, start, end, options) {
    await this.fetchLogstashStats(callCluster, monitoringClusterUuid, start, end, options);
    const allEphemeralIds = Object.values(options.allEphemeralIds).flat();
    if (allEphemeralIds.length > 0) {
      await this.fetchLogstashState(callCluster, monitoringClusterUuid, allEphemeralIds, options);
    }
    return options.clusters;
  }

  /*
   * Sets the index patterns based on the metricbeat monitoring types: [legacy, stack]
   * @param monitoringType - the monitoring type where metricbeat monitoring is intended.
   */
  setIndexPattern(monitoringType) {
    this.monitoringType = monitoringType;
    if (monitoringType === 'stack') {
      this.indexPattern.state = _constants.INDEX_PATTERN_LOGSTASH_STACK_MONITORING_STATE;
      this.indexPattern.stats = _constants.INDEX_PATTERN_LOGSTASH_STACK_MONITORING_STATS;
    } else {
      this.indexPattern.state = _constants.INDEX_PATTERN_LOGSTASH_MONITORING;
      this.indexPattern.stats = _constants.INDEX_PATTERN_LOGSTASH_MONITORING;
    }
  }
  getIndexPattern() {
    return this.indexPattern;
  }
  getMonitoringType() {
    return this.monitoringType;
  }

  /*
   * Update a clusters object with processed Logstash stats for metricbeat monitoring
   * @param {Array} results - array of LogstashStats docs from ES
   * @param {Object} clusters - LogstashBaseStats in an object keyed by the cluster UUIDs
   * @param {Object} allEphemeralIds - EphemeralIds in an object keyed by cluster UUIDs to track the pipelines for the cluster
   * @param {Object} versions - Versions in an object keyed by cluster UUIDs to track the logstash versions for the cluster
   * @param {Object} plugins - plugin information keyed by cluster UUIDs to count the unique plugins
   * @param {string} monitoringClusterUuid - monitoring cluster UUID
   */
  processStatsResults(results, {
    clusters,
    allEphemeralIds,
    versions,
    plugins
  }, monitoringClusterUuid) {
    var _results$hits;
    const currHits = (results === null || results === void 0 ? void 0 : (_results$hits = results.hits) === null || _results$hits === void 0 ? void 0 : _results$hits.hits) || [];
    currHits.forEach(hit => {
      var _logstash, _logstash$elasticsear, _logstash$elasticsear2, _hit$_source, _hit$_source$logstash, _hit$_source$logstash2;
      // consider orphan case as well
      // orphan case: where pipeline doesn't set cluster UUID or es-output plugin isn't in pipeline.
      const clusterUuid = ((_logstash = hit._source.logstash) === null || _logstash === void 0 ? void 0 : (_logstash$elasticsear = _logstash.elasticsearch) === null || _logstash$elasticsear === void 0 ? void 0 : (_logstash$elasticsear2 = _logstash$elasticsear.cluster) === null || _logstash$elasticsear2 === void 0 ? void 0 : _logstash$elasticsear2.id) || monitoringClusterUuid;
      if (clusterUuid !== undefined && clusters[clusterUuid] === undefined) {
        clusters[clusterUuid] = (0, _logstash_monitoring.getLogstashBaseStats)();
        versions[clusterUuid] = new Map();
        plugins[clusterUuid] = new Map();
      }
      const logstashStats = (_hit$_source = hit._source) === null || _hit$_source === void 0 ? void 0 : (_hit$_source$logstash = _hit$_source.logstash) === null || _hit$_source$logstash === void 0 ? void 0 : (_hit$_source$logstash2 = _hit$_source$logstash.node) === null || _hit$_source$logstash2 === void 0 ? void 0 : _hit$_source$logstash2.stats;
      if (clusterUuid !== undefined && logstashStats !== undefined) {
        var _logstashStats$logsta;
        const clusterStats = clusters[clusterUuid].cluster_stats || {};
        clusterStats.monitoringClusterUuid = monitoringClusterUuid;
        clusters[clusterUuid].count = (clusters[clusterUuid].count || 0) + 1;
        const thisVersion = (_logstashStats$logsta = logstashStats.logstash) === null || _logstashStats$logsta === void 0 ? void 0 : _logstashStats$logsta.version;
        const a = versions[clusterUuid];
        (0, _get_high_level_stats.incrementByKey)(a, thisVersion);
        clusters[clusterUuid].versions = (0, _get_high_level_stats.mapToList)(a, 'version');

        // for stack_monitoring, agent internally uses metricbeat or filebeat
        const thisCollectionType = this.getMonitoringType() === 'stack' ? 'agent' : 'metricbeat';
        if (!Object.hasOwn(clusterStats, 'collection_types')) {
          clusterStats.collection_types = {};
        }
        clusterStats.collection_types[thisCollectionType] = (clusterStats.collection_types[thisCollectionType] || 0) + 1;
        const pipelines = logstashStats.pipelines || [];
        pipelines.forEach(pipeline => {
          var _pipeline$queue;
          const thisQueueType = (_pipeline$queue = pipeline.queue) === null || _pipeline$queue === void 0 ? void 0 : _pipeline$queue.type;
          if (thisQueueType !== undefined) {
            if (!Object.hasOwn(clusterStats, 'queues')) {
              clusterStats.queues = {};
            }
            clusterStats.queues[thisQueueType] = (clusterStats.queues[thisQueueType] || 0) + 1;
          }
          const ephemeralId = pipeline.ephemeral_id;
          if (ephemeralId !== undefined) {
            allEphemeralIds[clusterUuid] = allEphemeralIds[clusterUuid] || [];
            allEphemeralIds[clusterUuid].push(ephemeralId);
          }
        });
      }
    });
  }

  /*
   * Update a clusters object with logstash state details
   * @param {Array} results - array of LogstashState docs from ES
   * @param {Object} clusters - LogstashBaseStats in an object keyed by the cluster UUIDs
   * @param {Object} plugins - plugin information keyed by cluster UUIDs to count the unique plugins
   * @param {string} monitoringClusterUuid - monitoring cluster UUID
   */
  processStateResults(results, {
    clusters,
    plugins
  }, monitoringClusterUuid) {
    var _results$hits2;
    const currHits = (results === null || results === void 0 ? void 0 : (_results$hits2 = results.hits) === null || _results$hits2 === void 0 ? void 0 : _results$hits2.hits) || [];
    currHits.forEach(hit => {
      var _hit$_source2, _hit$_source2$logstas, _hit$_source2$logstas2, _hit$_source2$logstas3, _clusters$clusterUuid, _clusters$clusterUuid2, _hit$_source3, _hit$_source3$logstas, _hit$_source3$logstas2, _hit$_source3$logstas3;
      const clusterUuid = ((_hit$_source2 = hit._source) === null || _hit$_source2 === void 0 ? void 0 : (_hit$_source2$logstas = _hit$_source2.logstash) === null || _hit$_source2$logstas === void 0 ? void 0 : (_hit$_source2$logstas2 = _hit$_source2$logstas.elasticsearch) === null || _hit$_source2$logstas2 === void 0 ? void 0 : (_hit$_source2$logstas3 = _hit$_source2$logstas2.cluster) === null || _hit$_source2$logstas3 === void 0 ? void 0 : _hit$_source2$logstas3.id) || monitoringClusterUuid;
      const pipelineStats = (_clusters$clusterUuid = clusters[clusterUuid]) === null || _clusters$clusterUuid === void 0 ? void 0 : (_clusters$clusterUuid2 = _clusters$clusterUuid.cluster_stats) === null || _clusters$clusterUuid2 === void 0 ? void 0 : _clusters$clusterUuid2.pipelines;
      const thisLogstashStatePipeline = (_hit$_source3 = hit._source) === null || _hit$_source3 === void 0 ? void 0 : (_hit$_source3$logstas = _hit$_source3.logstash) === null || _hit$_source3$logstas === void 0 ? void 0 : (_hit$_source3$logstas2 = _hit$_source3$logstas.node) === null || _hit$_source3$logstas2 === void 0 ? void 0 : (_hit$_source3$logstas3 = _hit$_source3$logstas2.state) === null || _hit$_source3$logstas3 === void 0 ? void 0 : _hit$_source3$logstas3.pipeline;
      if (pipelineStats !== undefined && thisLogstashStatePipeline !== undefined) {
        var _thisLogstashStatePip, _thisLogstashStatePip2, _clusters$clusterUuid3;
        pipelineStats.count = (pipelineStats.count || 0) + 1;
        const thisPipelineBatchSize = thisLogstashStatePipeline.batch_size;
        if (thisPipelineBatchSize !== undefined) {
          pipelineStats.batch_size_total = (pipelineStats.batch_size_total || 0) + thisPipelineBatchSize;
          pipelineStats.batch_size_max = pipelineStats.batch_size_max || 0;
          pipelineStats.batch_size_min = pipelineStats.batch_size_min || 0;
          pipelineStats.batch_size_avg = pipelineStats.batch_size_total / pipelineStats.count;
          if (thisPipelineBatchSize > pipelineStats.batch_size_max) {
            pipelineStats.batch_size_max = thisPipelineBatchSize;
          }
          if (pipelineStats.batch_size_min === 0 || thisPipelineBatchSize < pipelineStats.batch_size_min) {
            pipelineStats.batch_size_min = thisPipelineBatchSize;
          }
        }
        const thisPipelineWorkers = thisLogstashStatePipeline.workers;
        if (thisPipelineWorkers !== undefined) {
          pipelineStats.workers_total = (pipelineStats.workers_total || 0) + thisPipelineWorkers;
          pipelineStats.workers_max = pipelineStats.workers_max || 0;
          pipelineStats.workers_min = pipelineStats.workers_min || 0;
          pipelineStats.workers_avg = pipelineStats.workers_total / pipelineStats.count;
          if (thisPipelineWorkers > pipelineStats.workers_max) {
            pipelineStats.workers_max = thisPipelineWorkers;
          }
          if (pipelineStats.workers_min === 0 || thisPipelineWorkers < pipelineStats.workers_min) {
            pipelineStats.workers_min = thisPipelineWorkers;
          }
        }

        // Extract the vertices object from the pipeline representation. From this, we can
        // retrieve the source of the pipeline element on the configuration(from file, string, or
        // x-pack-config-management), and the input, filter and output plugins from that pipeline.
        const vertices = (_thisLogstashStatePip = thisLogstashStatePipeline.representation) === null || _thisLogstashStatePip === void 0 ? void 0 : (_thisLogstashStatePip2 = _thisLogstashStatePip.graph) === null || _thisLogstashStatePip2 === void 0 ? void 0 : _thisLogstashStatePip2.vertices;
        if (vertices !== undefined) {
          vertices.forEach(vertex => {
            var _vertex$meta, _vertex$meta$source;
            const configName = vertex.config_name;
            const pluginType = vertex.plugin_type;
            let pipelineConfig = (_vertex$meta = vertex.meta) === null || _vertex$meta === void 0 ? void 0 : (_vertex$meta$source = _vertex$meta.source) === null || _vertex$meta$source === void 0 ? void 0 : _vertex$meta$source.protocol;
            if (pipelineConfig !== undefined) {
              if (pipelineConfig === 'string' || pipelineConfig === 'str') {
                pipelineConfig = 'string';
              } else if (pipelineConfig === 'x-pack-config-management') {
                pipelineConfig = 'xpack';
              } else {
                pipelineConfig = 'file';
              }
              if (!Object.hasOwn(pipelineStats, 'sources')) {
                pipelineStats.sources = {};
              }
              pipelineStats.sources[pipelineConfig] = true;
            }
            if (configName !== undefined && pluginType !== undefined) {
              (0, _get_high_level_stats.incrementByKey)(plugins[clusterUuid], `logstash-${pluginType}-${configName}`);
            }
          });
        }
        const clusterStats = (_clusters$clusterUuid3 = clusters[clusterUuid]) === null || _clusters$clusterUuid3 === void 0 ? void 0 : _clusters$clusterUuid3.cluster_stats;
        if (clusterStats !== undefined) {
          clusterStats.plugins = (0, _get_high_level_stats.mapToList)(plugins[clusterUuid], 'name');
        }
      }
    });
  }

  /*
   * Creates a query and executes against ES to fetch metricbeat monitoring, Logstash stats metrics
   * @param {Object} callCluster - ES client
   * @param {string} monitoringClusterUuid - monitoring cluster UUID
   * @param {string} start - start timestamp
   * @param {string} end - end timestamp
   * @param {Object} options - additional processing required options
   */
  async fetchLogstashStats(callCluster, monitoringClusterUuid, start, end, {
    page = 0,
    ...options
  }) {
    var _results$hits3;
    const filterPath = ['hits.hits._source.cluster_uuid', 'hits.hits._source.type', 'hits.hits._source.source_node', 'hits.hits._source.agent.type', 'hits.hits._source.logstash.elasticsearch.cluster.id',
    // alias for cluster_uuid
    'hits.hits._source.logstash.node.stats.pipelines.id', 'hits.hits._source.logstash.node.stats.pipelines.ephemeral_id', 'hits.hits._source.logstash.node.stats.pipelines.queue.type', 'hits.hits._source.logstash.node.stats.logstash.version', 'hits.hits._source.logstash.node.stats.logstash.uuid'];
    const params = {
      index: this.indexPattern.stats,
      ignore_unavailable: true,
      filter_path: filterPath,
      query: (0, _create_query.createQuery)({
        start,
        end,
        filters: [{
          bool: {
            should: [{
              term: {
                'metricset.name': 'node_stats'
              }
            }, {
              term: {
                'data_stream.dataset': 'logstash.stack_monitoring.node_stats'
              }
            }]
          }
        }]
      }),
      collapse: {
        field: 'logstash.node.stats.logstash.uuid'
      },
      sort: [{
        ['timestamp']: {
          order: 'desc',
          unmapped_type: 'long'
        }
      }],
      from: page * _logstash_monitoring.HITS_SIZE,
      size: _logstash_monitoring.HITS_SIZE
    };
    const results = await callCluster.search(params, {
      headers: {
        'X-QUERY-SOURCE': _constants.TELEMETRY_QUERY_SOURCE
      }
    });
    const hitsLength = (results === null || results === void 0 ? void 0 : (_results$hits3 = results.hits) === null || _results$hits3 === void 0 ? void 0 : _results$hits3.hits.length) || 0;
    if (hitsLength > 0) {
      // further augment the clusters object with more stats
      this.processStatsResults(results, options, monitoringClusterUuid);
    }
    return Promise.resolve();
  }

  /*
   * Creates a query and executes against ES to fetch metricbeat monitoring, Logstash state metrics
   * @param {Object} callCluster - ES client
   * @param {string} monitoringClusterUuid - monitoring cluster UUID
   * @param {Array} ephemeralIds - Logstash pipeline ephemeral IDs
   * @param {string} start - start timestamp
   * @param {string} end - end timestamp
   * @param {Object} options - additional processing required options
   */
  async fetchLogstashState(callCluster, monitoringClusterUuid, ephemeralIds, {
    page = 0,
    ...options
  }) {
    var _results$hits4;
    const filterPath = ['hits.hits._source.logstash.node.state.pipeline.batch_size', 'hits.hits._source.logstash.node.state.pipeline.workers', 'hits.hits._source.logstash.node.state.pipeline.representation.graph.vertices', 'hits.hits._source.type'];
    const params = {
      index: this.indexPattern.state,
      ignore_unavailable: true,
      filter_path: filterPath,
      query: (0, _create_query.createQuery)({
        // metricbeat occasionally sends state metrics
        // so, not using start and end periods as we need node state info to fill plugin usages
        filters: [{
          terms: {
            'logstash.node.state.pipeline.ephemeral_id': ephemeralIds
          }
        }, {
          bool: {
            should: [{
              term: {
                'metricset.name': 'node'
              }
            }, {
              term: {
                'data_stream.dataset': 'logstash.stack_monitoring.node'
              }
            }]
          }
        }]
      }),
      collapse: {
        field: 'logstash.node.state.pipeline.ephemeral_id'
      },
      sort: [{
        ['timestamp']: {
          order: 'desc',
          unmapped_type: 'long'
        }
      }],
      size: ephemeralIds.length
    };
    const results = await callCluster.search(params, {
      headers: {
        'X-QUERY-SOURCE': _constants.TELEMETRY_QUERY_SOURCE
      }
    });
    const hitsLength = (results === null || results === void 0 ? void 0 : (_results$hits4 = results.hits) === null || _results$hits4 === void 0 ? void 0 : _results$hits4.hits.length) || 0;
    if (hitsLength > 0) {
      // further augment the clusters object with more stats
      this.processStateResults(results, options, monitoringClusterUuid);
    }
    return Promise.resolve();
  }
}
exports.LogstashMetricbeatMonitoring = LogstashMetricbeatMonitoring;