"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerContextFunction = registerContextFunction;
exports.removeContextToolRequest = removeContextToolRequest;
var _gptTokenizer = require("gpt-tokenizer");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _common = require("../../../common");
var _create_function_response_message = require("../../../common/utils/create_function_response_message");
var _recall_and_score = require("./utils/recall_and_score");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_TOKEN_COUNT_FOR_DATA_ON_SCREEN = 1000;
function registerContextFunction({
  client,
  functions,
  resources,
  scopes,
  isKnowledgeBaseReady
}) {
  functions.registerFunction({
    name: _common.CONTEXT_FUNCTION_NAME,
    description: 'This function provides context as to what the user is looking at on their screen, and recalled documents from the knowledge base that matches their query',
    isInternal: true
  }, async ({
    connectorId,
    messages,
    screenContexts,
    chat
  }, signal) => {
    const {
      request,
      plugins
    } = resources;
    const {
      analytics
    } = await plugins.core.start();
    const actionsClient = await (await plugins.actions.start()).getActionsClientWithRequest(request);
    async function getContext() {
      const screenDescription = (0, _lodash.compact)(screenContexts.map(context => context.screenDescription)).join('\n\n');

      // any data that falls within the token limit, send it automatically
      const dataWithinTokenLimit = (0, _lodash.compact)(screenContexts.flatMap(context => context.data)).filter(data => (0, _gptTokenizer.encode)(JSON.stringify(data.value)).length <= MAX_TOKEN_COUNT_FOR_DATA_ON_SCREEN);
      const content = {
        screen_description: screenDescription,
        learnings: [],
        ...(dataWithinTokenLimit.length ? {
          data_on_screen: dataWithinTokenLimit
        } : {})
      };
      if (!isKnowledgeBaseReady) {
        resources.logger.warn('Knowledge base is not ready yet. Returning context function response without knowledge base entries.');
        return {
          content
        };
      }
      const connector = await actionsClient.get({
        id: connectorId,
        throwIfSystemAction: true
      });
      const {
        llmScores,
        relevantDocuments,
        suggestions
      } = await (0, _recall_and_score.recallAndScore)({
        recall: client.recall,
        chat,
        logger: resources.logger,
        screenDescription,
        messages: removeContextToolRequest(messages),
        signal,
        analytics,
        scopes,
        connector
      });
      return {
        content: {
          ...content,
          learnings: relevantDocuments
        },
        data: {
          llmScores,
          suggestions
        }
      };
    }
    return new _rxjs.Observable(subscriber => {
      getContext().then(({
        content,
        data
      }) => {
        subscriber.next((0, _create_function_response_message.createFunctionResponseMessage)({
          name: _common.CONTEXT_FUNCTION_NAME,
          content,
          data
        }));
        subscriber.complete();
      }).catch(error => {
        resources.logger.error('Error in context function');
        resources.logger.error(error);
        subscriber.next((0, _create_function_response_message.createFunctionResponseMessage)({
          name: _common.CONTEXT_FUNCTION_NAME,
          content: `Error in context function: ${error.message}`,
          data: {
            error: {
              message: error.message
            }
          }
        }));
        subscriber.complete();
      });
    });
  });
}
function removeContextToolRequest(messages) {
  var _lastMessage$message$;
  const lastMessage = (0, _lodash.last)(messages);
  if ((lastMessage === null || lastMessage === void 0 ? void 0 : (_lastMessage$message$ = lastMessage.message.function_call) === null || _lastMessage$message$ === void 0 ? void 0 : _lastMessage$message$.name) === _common.CONTEXT_FUNCTION_NAME) {
    return messages.slice(0, -1);
  }
  return messages;
}