"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deserializeCluster = deserializeCluster;
exports.serializeCluster = serializeCluster;
var _validate_address = require("./validate_address");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Values returned from ES GET /_remote/info
/**
 * TODO: This interface needs to be updated with values from {@link RemoteInfo} provided
 * by the @elastic/elasticsearch client
 */

// Payload expected from ES PUT /_cluster/settings

function deserializeCluster(name, esClusterObject, deprecatedProxyAddress, isCloudEnabled, nodeConnectionsSettings, proxySocketConnectionsSettings) {
  var _extractHostAndPort;
  if (!name || !esClusterObject || typeof esClusterObject !== 'object') {
    throw new Error('Unable to deserialize cluster');
  }
  const {
    seeds,
    mode,
    connected: isConnected,
    num_nodes_connected: connectedNodesCount,
    max_connections_per_cluster: maxConnectionsPerCluster,
    initial_connect_timeout: initialConnectTimeout,
    skip_unavailable: skipUnavailable,
    transport,
    proxy_address: proxyAddress,
    max_proxy_socket_connections: maxProxySocketConnections,
    num_proxy_sockets_connected: connectedSocketsCount,
    server_name: serverName,
    cluster_credentials: clusterCredentials,
    node_connections: nodeConnections,
    proxy_socket_connections: proxySocketConnections
  } = esClusterObject;
  let deserializedClusterObject = {
    name,
    mode,
    isConnected,
    connectedNodesCount,
    maxConnectionsPerCluster,
    initialConnectTimeout,
    skipUnavailable,
    seeds,
    proxyAddress,
    proxySocketConnections,
    maxProxySocketConnections,
    connectedSocketsCount,
    serverName,
    nodeConnections,
    securityModel: clusterCredentials ? _constants.SECURITY_MODEL.API : _constants.SECURITY_MODEL.CERTIFICATE
  };
  if (mode === _constants.SNIFF_MODE && !nodeConnections) {
    deserializedClusterObject.nodeConnections = nodeConnectionsSettings || null;
  }
  if (mode === _constants.PROXY_MODE && !proxySocketConnections) {
    deserializedClusterObject.proxySocketConnections = proxySocketConnectionsSettings || null;
  }
  if (transport) {
    const {
      ping_schedule: transportPingSchedule,
      compress: transportCompress
    } = transport;
    deserializedClusterObject = {
      ...deserializedClusterObject,
      transportPingSchedule,
      transportCompress
    };
  }
  const deprecatedProxyHost = deprecatedProxyAddress ? (_extractHostAndPort = (0, _validate_address.extractHostAndPort)(deprecatedProxyAddress)) === null || _extractHostAndPort === void 0 ? void 0 : _extractHostAndPort.host : undefined;

  // If a user has a remote cluster with the deprecated proxy setting,
  // we transform the data to support the new implementation and also flag the deprecation
  if (deprecatedProxyAddress && deprecatedProxyHost) {
    deserializedClusterObject = {
      ...deserializedClusterObject,
      proxyAddress: deprecatedProxyAddress,
      seeds: undefined,
      hasDeprecatedProxySetting: true,
      mode: _constants.PROXY_MODE,
      // Cloud-specific logic: Create default server name, since this field doesn't exist in deprecated implementation
      serverName: isCloudEnabled ? deprecatedProxyHost : undefined
    };
  }

  // It's unnecessary to send undefined values back to the client, so we can remove them.
  Object.keys(deserializedClusterObject).forEach(key => {
    if (deserializedClusterObject[key] === undefined) {
      delete deserializedClusterObject[key];
    }
  });
  return deserializedClusterObject;
}
function serializeCluster(deserializedClusterObject, previousClusterMode, isDelete) {
  if (!deserializedClusterObject || typeof deserializedClusterObject !== 'object') {
    throw new Error('Unable to serialize cluster');
  }
  const {
    name,
    seeds,
    skipUnavailable,
    mode,
    nodeConnections,
    proxyAddress,
    proxySocketConnections,
    serverName,
    hasDeprecatedProxySetting
  } = deserializedClusterObject;
  const clusterData = {
    skip_unavailable: typeof skipUnavailable === 'boolean' ? skipUnavailable : null,
    mode: mode || null,
    ...(mode === _constants.PROXY_MODE ? {
      proxy_address: proxyAddress || null,
      proxy_socket_connections: proxySocketConnections || null,
      server_name: serverName || null
    } : {
      seeds: seeds || null,
      node_connections: nodeConnections || null
    })
  };

  // If the cluster is been deleted, we need to set all values to null
  // If the cluster is been edited and the mode has changed, we need to set to null the previous mode settings values
  if (isDelete || previousClusterMode && previousClusterMode !== mode) {
    clusterData.proxy_address = proxyAddress || null;
    clusterData.proxy_socket_connections = proxySocketConnections || null;
    clusterData.server_name = serverName || null;
    clusterData.seeds = seeds || null;
    clusterData.node_connections = nodeConnections || null;
  }

  // This is only applicable in edit mode
  // In order to "upgrade" an existing remote cluster to use the new proxy mode settings, we need to set the old proxy setting to null
  if (hasDeprecatedProxySetting) {
    clusterData.proxy = null;
  }
  return {
    // Background on why we only save as persistent settings detailed here: https://github.com/elastic/kibana/pull/26067#issuecomment-441848124
    persistent: {
      cluster: {
        remote: {
          [name]: clusterData
        }
      }
    }
  };
}