"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.resultToFieldFromMappings = exports.resultToFieldFromMappingResponse = exports.resultTitle = exports.resultMetaData = exports.reorderFieldsInImportance = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TITLE_KEYS = ['title', 'name'];
const isRecord = source => {
  return typeof source === 'object' && source !== null;
};
function hasStringKey(source, key) {
  return typeof source[key] === 'string';
}
const resultTitle = result => {
  if (isRecord(result._source)) {
    for (const key of TITLE_KEYS) {
      if (hasStringKey(result._source, key)) {
        return result._source[key];
      }
    }
  }
  return undefined;
};
exports.resultTitle = resultTitle;
const resultMetaData = result => ({
  id: result._id,
  title: resultTitle(result),
  score: result._score
});
exports.resultMetaData = resultMetaData;
const MAPPING_TYPE_ORDER = ['semantic_text', 'dense_vector', 'sparse_vector'];
const SPECIAL_NAME_FIELDS = ['headings'];

/**
 * Reorders an array of fields based on their importance.
 *
 * The function sorts the fields by checking if their names are in the `SPECIAL_NAME_FIELDS` array first and then by
 * their mapping type (semantic_text, dense_vector, sparse_vector) if they are not in the `SPECIAL_NAME_FIELDS` array.
 *
 * @param fields - An array of field properties to be reordered.
 * @returns The reordered array of field properties.
 */
const reorderFieldsInImportance = fields => {
  return fields.sort((a, b) => {
    const specialA = SPECIAL_NAME_FIELDS.indexOf(a.fieldName);
    const specialB = SPECIAL_NAME_FIELDS.indexOf(b.fieldName);
    if (specialA !== -1 || specialB !== -1) {
      if (specialA === -1) return 1;
      if (specialB === -1) return -1;
      return specialA - specialB;
    }
    const typeA = MAPPING_TYPE_ORDER.indexOf(a.fieldType);
    const typeB = MAPPING_TYPE_ORDER.indexOf(b.fieldType);
    const orderA = typeA === -1 ? MAPPING_TYPE_ORDER.length : typeA;
    const orderB = typeB === -1 ? MAPPING_TYPE_ORDER.length : typeB;
    if (orderA === orderB) {
      return a.fieldName.localeCompare(b.fieldName);
    }
    return orderA - orderB;
  });
};
exports.reorderFieldsInImportance = reorderFieldsInImportance;
const resultToFieldFromMappingResponse = (result, mappings) => {
  if (mappings && mappings[result._index] && result._source && !Array.isArray(result._source)) {
    if (typeof result._source === 'object') {
      return Object.entries(result._source).map(([key, value]) => {
        var _mappings$result$_ind, _mappings$result$_ind2, _mappings$result$_ind3, _mappings$result$_ind4, _mappings$result$_ind5;
        return {
          fieldName: key,
          fieldType: (_mappings$result$_ind = (_mappings$result$_ind2 = mappings[result._index]) === null || _mappings$result$_ind2 === void 0 ? void 0 : (_mappings$result$_ind3 = _mappings$result$_ind2.mappings) === null || _mappings$result$_ind3 === void 0 ? void 0 : (_mappings$result$_ind4 = _mappings$result$_ind3.properties) === null || _mappings$result$_ind4 === void 0 ? void 0 : (_mappings$result$_ind5 = _mappings$result$_ind4[key]) === null || _mappings$result$_ind5 === void 0 ? void 0 : _mappings$result$_ind5.type) !== null && _mappings$result$_ind !== void 0 ? _mappings$result$_ind : 'object',
          fieldValue: JSON.stringify(value, null, 2)
        };
      });
    }
  }
  return [];
};
exports.resultToFieldFromMappingResponse = resultToFieldFromMappingResponse;
const resultToFieldFromMappings = (result, mappings) => {
  if (mappings && result._source && !Array.isArray(result._source)) {
    return Object.entries(result._source).map(([key, value]) => {
      var _mappings$key$type, _mappings$key;
      return {
        fieldName: key,
        fieldType: (_mappings$key$type = (_mappings$key = mappings[key]) === null || _mappings$key === void 0 ? void 0 : _mappings$key.type) !== null && _mappings$key$type !== void 0 ? _mappings$key$type : 'object',
        fieldValue: JSON.stringify(value, null, 2)
      };
    });
  }
  return [];
};
exports.resultToFieldFromMappings = resultToFieldFromMappings;