"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UiamService = void 0;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _fs = require("fs");
var _undici = require("undici");
var _ = require("..");
var _constants = require("../../common/constants");
var _errors = require("../errors");
function _classPrivateMethodInitSpec(e, a) { _checkPrivateRedeclaration(e, a), a.add(e); }
function _classPrivateFieldInitSpec(e, t, a) { _checkPrivateRedeclaration(e, t), t.set(e, a); }
function _checkPrivateRedeclaration(e, t) { if (t.has(e)) throw new TypeError("Cannot initialize the same private elements twice on an object"); }
function _classPrivateFieldGet(s, a) { return s.get(_assertClassBrand(s, a)); }
function _classPrivateFieldSet(s, a, r) { return s.set(_assertClassBrand(s, a), r), r; }
function _assertClassBrand(e, t, n) { if ("function" == typeof e ? e === t : e.has(t)) return arguments.length < 3 ? t : n; throw new TypeError("Private element is not present on this object"); } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
/**
 * The service that integrates with UIAM for user authentication and session management.
 */
var _logger = /*#__PURE__*/new WeakMap();
var _config = /*#__PURE__*/new WeakMap();
var _dispatcher = /*#__PURE__*/new WeakMap();
var _UiamService_brand = /*#__PURE__*/new WeakSet();
/**
 * See {@link UiamServicePublic}.
 */
class UiamService {
  constructor(logger, config) {
    /**
     * Creates a custom dispatcher for the native `fetch` to use custom TLS connection settings.
     */
    _classPrivateMethodInitSpec(this, _UiamService_brand);
    _classPrivateFieldInitSpec(this, _logger, void 0);
    _classPrivateFieldInitSpec(this, _config, void 0);
    _classPrivateFieldInitSpec(this, _dispatcher, void 0);
    _classPrivateFieldSet(_logger, this, logger);

    // Destructure existing config and re-create it again after validation to make TypeScript can infer the proper types.
    const {
      enabled,
      url,
      sharedSecret,
      ssl
    } = config;
    if (!enabled) {
      throw new Error('UIAM is not enabled.');
    }
    if (!url) {
      throw new Error('UIAM URL is not configured.');
    }
    if (!sharedSecret) {
      throw new Error('UIAM shared secret is not configured.');
    }
    _classPrivateFieldSet(_config, this, {
      enabled,
      url,
      sharedSecret,
      ssl
    });
    _classPrivateFieldSet(_dispatcher, this, _assertClassBrand(_UiamService_brand, this, _createFetchDispatcher).call(this));
  }

  /**
   * See {@link UiamServicePublic.getAuthenticationHeaders}.
   */
  getAuthenticationHeaders(accessToken) {
    return {
      authorization: new _.HTTPAuthorizationHeader('Bearer', accessToken).toString(),
      [_constants.ES_CLIENT_AUTHENTICATION_HEADER]: _classPrivateFieldGet(_config, this).sharedSecret
    };
  }

  /**
   * See {@link UiamServicePublic.getUserProfileGrant}.
   */
  getUserProfileGrant(accessToken) {
    return {
      type: 'uiamAccessToken',
      accessToken,
      sharedSecret: _classPrivateFieldGet(_config, this).sharedSecret
    };
  }

  /**
   * See {@link UiamServicePublic.refreshSessionTokens}.
   */
  async refreshSessionTokens(refreshToken) {
    try {
      const tokens = await _parseUiamResponse.call(UiamService, await fetch(`${_classPrivateFieldGet(_config, this).url}/uiam/api/v1/tokens/_refresh`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          [_constants.ES_CLIENT_AUTHENTICATION_HEADER]: _classPrivateFieldGet(_config, this).sharedSecret
        },
        body: JSON.stringify({
          refresh_token: refreshToken
        }),
        // @ts-expect-error Undici `fetch` supports `dispatcher` option, see https://github.com/nodejs/undici/pull/1411.
        dispatcher: _classPrivateFieldGet(_dispatcher, this)
      }));
      return {
        accessToken: tokens.access_token,
        refreshToken: tokens.refresh_token
      };
    } catch (err) {
      // TODO: Temporarily rethrow all errors as `401` (UIAM service doesn't support token refresh yet).
      _classPrivateFieldGet(_logger, this).error(() => `Failed to refresh session tokens: ${(0, _errors.getDetailedErrorMessage)(err)}`);
      throw _boom.default.unauthorized(`UIAM service doesn't support token refresh yet.`);
    }
  }

  /**
   * See {@link UiamServicePublic.invalidateSessionTokens}.
   */
  async invalidateSessionTokens(accessToken, refreshToken) {
    let invalidatedTokensCount;
    try {
      invalidatedTokensCount = (await _parseUiamResponse.call(UiamService, await fetch(`${_classPrivateFieldGet(_config, this).url}/uiam/api/v1/tokens/_invalidate`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          Authorization: `Bearer ${accessToken}`
        },
        body: JSON.stringify({
          token: accessToken,
          refresh_token: refreshToken
        }),
        // @ts-expect-error Undici `fetch` supports `dispatcher` option, see https://github.com/nodejs/undici/pull/1411.
        dispatcher: _classPrivateFieldGet(_dispatcher, this)
      }))).invalidated_tokens;
    } catch (err) {
      _classPrivateFieldGet(_logger, this).error(() => `Failed to invalidate session tokens: ${(0, _errors.getDetailedErrorMessage)(err)}`);

      // TODO: Temporarily swallow the 500 errors (UIAM service doesn't support token invalidation yet).
      if ((0, _errors.getErrorStatusCode)(err) === 500) {
        return;
      }
      throw err;
    }
    if (invalidatedTokensCount === 0) {
      _classPrivateFieldGet(_logger, this).debug('Session tokens were already invalidated.');
    } else if (invalidatedTokensCount === 2) {
      _classPrivateFieldGet(_logger, this).debug('All session tokens were successfully invalidated.');
    } else {
      _classPrivateFieldGet(_logger, this).warn(`${invalidatedTokensCount} refresh tokens were invalidated, this is unexpected.`);
    }
  }
}
exports.UiamService = UiamService;
function _createFetchDispatcher() {
  const {
    certificateAuthorities,
    verificationMode
  } = _classPrivateFieldGet(_config, this).ssl;

  // Read CA certificate(s) from the file paths defined in the config.
  const ca = certificateAuthorities ? (Array.isArray(certificateAuthorities) ? certificateAuthorities : [certificateAuthorities]).map(caPath => (0, _fs.readFileSync)(caPath, 'utf8')) : undefined;

  // If we don't have custom CAs and the full verification is required, we don't need custom
  // dispatcher as it's a default `fetch` behavior.
  if (!ca && verificationMode === 'full') {
    return;
  }
  return new _undici.Agent({
    connect: {
      ca,
      rejectUnauthorized: verificationMode !== 'none',
      // By default, Node.js is checking the server identity to match SAN/CN in certificate.
      ...(verificationMode === 'certificate' ? {
        checkServerIdentity: () => undefined
      } : {})
    }
  });
}
/**
 * Parses the UIAM service response as free-form JSON if it’s a successful response, otherwise throws a Boom error based on the error response from the UIAM service.
 */
async function _parseUiamResponse(response) {
  var _payload$error;
  if (response.ok) {
    return await response.json();
  }
  const payload = await response.json();
  const err = new _boom.default.Boom((payload === null || payload === void 0 ? void 0 : (_payload$error = payload.error) === null || _payload$error === void 0 ? void 0 : _payload$error.message) || 'Unknown error');
  err.output = {
    statusCode: response.status,
    payload,
    headers: Object.fromEntries(response.headers.entries())
  };
  throw err;
}