"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.threatMatchMappingValidatorFactory = threatMatchMappingValidatorFactory;
var _i18n = require("@kbn/i18n");
var _helpers = require("../../../../../common/components/threat_match/helpers");
var _error_codes = require("./error_codes");
var _get_unknown_threat_match_mapping_field_names = require("./get_unknown_threat_match_mapping_field_names");
var _indicator_match = require("../../../../../../common/utils/request_validation/indicator_match");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function threatMatchMappingValidatorFactory({
  indexPatterns,
  threatIndexPatterns
}) {
  return (...args) => {
    const [{
      path,
      value
    }] = args;
    if ((0, _helpers.singleEntryThreat)(value)) {
      return {
        code: _error_codes.THREAT_MATCH_MAPPING_ERROR_CODES.ERR_FIELD_MISSING,
        path,
        message: _i18n.i18n.translate('xpack.securitySolution.detectionEngine.ruleManagement.createRule.threatMappingField.requiredError', {
          defaultMessage: 'At least one indicator match is required.'
        })
      };
    }
    if ((0, _helpers.containsInvalidItems)(value)) {
      return {
        code: _error_codes.THREAT_MATCH_MAPPING_ERROR_CODES.ERR_FIELD_MISSING,
        path,
        message: _i18n.i18n.translate('xpack.securitySolution.detectionEngine.ruleManagement.threatMappingField.bothFieldNamesRequiredError', {
          defaultMessage: 'All matches require both a field and threat index field.'
        })
      };
    }
    const {
      unknownSourceIndicesFields,
      unknownThreatMatchIndicesFields
    } = (0, _get_unknown_threat_match_mapping_field_names.getUnknownThreatMatchMappingFieldNames)({
      entries: value,
      indexPatterns,
      threatIndexPatterns
    });
    if (unknownSourceIndicesFields.length > 0 && unknownThreatMatchIndicesFields.length > 0) {
      return {
        code: _error_codes.THREAT_MATCH_MAPPING_ERROR_CODES.ERR_FIELDS_UNKNOWN,
        path,
        message: _i18n.i18n.translate('xpack.securitySolution.detectionEngine.ruleManagement.threatMappingField.unknownFields', {
          defaultMessage: 'Indicator mapping has unknown fields. {unknownSourceIndicesFields} fields not found in the source events indices and {unknownThreatMatchIndicesFields} fields not found in the indicator indices.',
          values: {
            unknownSourceIndicesFields: `"${unknownSourceIndicesFields.join('", "')}"`,
            unknownThreatMatchIndicesFields: `"${unknownThreatMatchIndicesFields.join('", "')}"`
          }
        })
      };
    }
    if (unknownSourceIndicesFields.length > 0) {
      return {
        code: _error_codes.THREAT_MATCH_MAPPING_ERROR_CODES.ERR_FIELDS_UNKNOWN,
        path,
        message: _i18n.i18n.translate('xpack.securitySolution.detectionEngine.ruleManagement.threatMappingField.unknownSourceIndicesFields', {
          defaultMessage: 'Indicator mapping has unknown fields. {unknownSourceIndicesFields} fields not found in the source events indices.',
          values: {
            unknownSourceIndicesFields: `"${unknownSourceIndicesFields.join('", "')}"`
          }
        })
      };
    }
    if (unknownThreatMatchIndicesFields.length > 0) {
      return {
        code: _error_codes.THREAT_MATCH_MAPPING_ERROR_CODES.ERR_FIELDS_UNKNOWN,
        path,
        message: _i18n.i18n.translate('xpack.securitySolution.detectionEngine.ruleManagement.threatMappingField.unknownIndicatorIndicesFields', {
          defaultMessage: 'Indicator mapping has unknown fields. {unknownThreatMatchIndicesFields} fields not found in the indicator indices.',
          values: {
            unknownThreatMatchIndicesFields: `"${unknownThreatMatchIndicesFields.join('", "')}"`
          }
        })
      };
    }
    if ((0, _indicator_match.containsDoesNotMatchEntriesOnly)(value)) {
      return {
        code: _error_codes.THREAT_MATCH_MAPPING_ERROR_CODES.ERR_SINGLE_NOT_MATCH_CLAUSE,
        path,
        message: _i18n.i18n.translate('xpack.securitySolution.detectionEngine.ruleManagement.threatMappingField.singleNotMatchClauseError', {
          defaultMessage: 'Entries with AND clauses must have at least one MATCHES condition.'
        })
      };
    }
    if ((0, _indicator_match.containsInvalidDoesNotMatchEntries)(value)) {
      return {
        code: _error_codes.THREAT_MATCH_MAPPING_ERROR_CODES.ERR_INVALID_NOT_MATCH_CLAUSE,
        path,
        message: _i18n.i18n.translate('xpack.securitySolution.detectionEngine.ruleManagement.threatMappingField.invalidNotMatchClauseError', {
          defaultMessage: 'DOES NOT MATCH and MATCHES entries that are connected by an AND clause cannot use the same threat mappings. Choose a different threat mapping for one of the entries.'
        })
      };
    }
  };
}