"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFieldBrowserOptions = void 0;
var _react = require("react");
var _reactRedux = require("react-redux");
var _use_experimental_features = require("../../../common/hooks/use_experimental_features");
var _use_data_view = require("../../../data_view_manager/hooks/use_data_view");
var _use_data_view2 = require("../../../common/containers/source/use_data_view");
var _kibana = require("../../../common/lib/kibana");
var _store = require("../../../common/store");
var _default_headers = require("../timeline/body/column_headers/default_headers");
var _constants = require("../timeline/body/constants");
var _create_field_button = require("./create_field_button");
var _field_table_columns = require("./field_table_columns");
var _use_start_transaction = require("../../../common/lib/apm/use_start_transaction");
var _user_actions = require("../../../common/lib/apm/user_actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This hook is used in the alerts table and explore page tables (StatefulEventsViewer) to manage field browser options.
 */
const useFieldBrowserOptions = ({
  sourcererScope,
  editorActionsRef,
  removeColumn,
  upsertColumn
}) => {
  const newDataViewPickerEnabled = (0, _use_experimental_features.useIsExperimentalFeatureEnabled)('newDataViewPickerEnabled');
  const [dataView, setDataView] = (0, _react.useState)(null);
  const {
    dataView: experimentalDataView
  } = (0, _use_data_view.useDataView)(sourcererScope);
  const {
    startTransaction
  } = (0, _use_start_transaction.useStartTransaction)();
  const {
    indexFieldsSearch
  } = (0, _use_data_view2.useDataView)();
  const {
    dataViewFieldEditor,
    data: {
      dataViews
    }
  } = (0, _kibana.useKibana)().services;
  const missingPatterns = (0, _reactRedux.useSelector)(state => {
    return _store.sourcererSelectors.sourcererScopeMissingPatterns(state, sourcererScope);
  });
  const sourcererDataViewId = (0, _reactRedux.useSelector)(state => {
    return _store.sourcererSelectors.sourcererScopeSelectedDataViewId(state, sourcererScope);
  });
  const selectedDataViewId = (0, _react.useMemo)(() => newDataViewPickerEnabled ? experimentalDataView === null || experimentalDataView === void 0 ? void 0 : experimentalDataView.id : sourcererDataViewId, [sourcererDataViewId, experimentalDataView === null || experimentalDataView === void 0 ? void 0 : experimentalDataView.id, newDataViewPickerEnabled]);
  (0, _react.useEffect)(() => {
    let ignore = false;
    const fetchAndSetDataView = async dataViewId => {
      if (newDataViewPickerEnabled) {
        if (experimentalDataView) setDataView(experimentalDataView);
        return;
      }
      const aDatView = await dataViews.get(dataViewId);
      if (ignore) return;
      setDataView(aDatView);
    };
    if (selectedDataViewId != null && !missingPatterns.length) {
      fetchAndSetDataView(selectedDataViewId);
    }
    return () => {
      ignore = true;
    };
  }, [selectedDataViewId, missingPatterns, dataViews, newDataViewPickerEnabled, experimentalDataView]);
  const openFieldEditor = (0, _react.useCallback)(async fieldName => {
    if (dataView && selectedDataViewId) {
      const closeFieldEditor = await dataViewFieldEditor.openEditor({
        ctx: {
          dataView
        },
        fieldName,
        onSave: async savedFields => {
          startTransaction({
            name: _user_actions.FIELD_BROWSER_ACTIONS.FIELD_SAVED
          });
          // Fetch the updated list of fields
          // Using cleanCache since the number of fields might have not changed, but we need to update the state anyway
          if (newDataViewPickerEnabled) {
            dataViews.clearInstanceCache(selectedDataViewId);
          } else {
            await indexFieldsSearch({
              dataViewId: selectedDataViewId,
              cleanCache: true
            });
          }
          for (const savedField of savedFields) {
            if (fieldName && fieldName !== savedField.name) {
              // Remove old field from event table when renaming a field
              removeColumn(fieldName);
            }

            // Add the saved column field to the table in any case

            upsertColumn({
              columnHeaderType: _default_headers.defaultColumnHeaderType,
              id: savedField.name,
              initialWidth: _constants.DEFAULT_COLUMN_MIN_WIDTH
            }, 0);
          }
          if (editorActionsRef) {
            editorActionsRef.current = null;
          }
        }
      });
      if (editorActionsRef) {
        editorActionsRef.current = {
          closeEditor: () => {
            editorActionsRef.current = null;
            closeFieldEditor();
          }
        };
      }
    }
  }, [dataView, selectedDataViewId, dataViewFieldEditor, editorActionsRef, startTransaction, newDataViewPickerEnabled, dataViews, indexFieldsSearch, upsertColumn, removeColumn]);
  const openDeleteFieldModal = (0, _react.useCallback)(fieldName => {
    if (dataView && selectedDataViewId) {
      dataViewFieldEditor.openDeleteModal({
        ctx: {
          dataView
        },
        fieldName,
        onDelete: async () => {
          startTransaction({
            name: _user_actions.FIELD_BROWSER_ACTIONS.FIELD_DELETED
          });
          if (newDataViewPickerEnabled) {
            dataViews.clearInstanceCache(selectedDataViewId);
          } else {
            await indexFieldsSearch({
              dataViewId: selectedDataViewId,
              cleanCache: true
            });
          }
          removeColumn(fieldName);
        }
      });
    }
  }, [dataView, selectedDataViewId, dataViewFieldEditor, startTransaction, newDataViewPickerEnabled, removeColumn, dataViews, indexFieldsSearch]);
  const hasFieldEditPermission = (0, _react.useMemo)(() => dataViewFieldEditor === null || dataViewFieldEditor === void 0 ? void 0 : dataViewFieldEditor.userPermissions.editIndexPattern(), [dataViewFieldEditor === null || dataViewFieldEditor === void 0 ? void 0 : dataViewFieldEditor.userPermissions]);
  const createFieldButton = (0, _create_field_button.useCreateFieldButton)({
    isAllowed: hasFieldEditPermission && !!selectedDataViewId && !(dataView !== null && dataView !== void 0 && dataView.managed),
    loading: !dataView,
    openFieldEditor
  });
  const getFieldTableColumns = (0, _field_table_columns.useFieldTableColumns)({
    hasFieldEditPermission,
    openFieldEditor,
    openDeleteFieldModal
  });
  const memoized = (0, _react.useMemo)(() => ({
    createFieldButton,
    getFieldTableColumns
  }), [createFieldButton, getFieldTableColumns]);
  return memoized;
};
exports.useFieldBrowserOptions = useFieldBrowserOptions;