"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.resolveTimeline = exports.persistTimeline = exports.persistFavorite = exports.installPrepackedTimelines = exports.importTimelines = exports.getTimelineTemplate = exports.getTimeline = exports.getDraftTimeline = exports.getAllTimelines = exports.exportSelectedTimeline = exports.deleteTimelinesByIds = exports.copyTimeline = exports.cleanDraftTimeline = void 0;
var _lodash = require("lodash");
var _timeline = require("../../../common/api/timeline");
var _constants = require("../../../common/constants");
var _kibana = require("../../common/lib/kibana");
var _toasters = require("../../common/components/toasters");
var _zod_errors = require("../../../common/timelines/zod_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createToasterPlainError = message => new _toasters.ToasterError([message]);
const parseOrThrow = (0, _zod_errors.parseOrThrowErrorFactory)(createToasterPlainError);
const decodeTimelineResponse = respTimeline => parseOrThrow(_timeline.PersistTimelineResponse)(respTimeline);
const decodeSingleTimelineResponse = respTimeline => parseOrThrow(_timeline.GetTimelineResponse)(respTimeline);
const decodeResolvedSingleTimelineResponse = respTimeline => parseOrThrow(_timeline.ResolveTimelineResponse)(respTimeline);
const decodeGetTimelinesResponse = respTimeline => parseOrThrow(_timeline.GetTimelinesResponse)(respTimeline);
const decodeTimelineErrorResponse = respTimeline => parseOrThrow(_timeline.TimelineErrorResponse)(respTimeline);
const decodePrepackedTimelineResponse = respTimeline => parseOrThrow(_timeline.ImportTimelineResult)(respTimeline);
const decodeResponseFavoriteTimeline = respTimeline => parseOrThrow(_timeline.PersistFavoriteRouteResponse)(respTimeline);
const postTimeline = async ({
  timeline
}) => {
  let requestBody;
  try {
    requestBody = JSON.stringify({
      timeline
    });
  } catch (err) {
    return Promise.reject(new Error(`Failed to stringify query: ${JSON.stringify(err)}`));
  }
  const response = await _kibana.KibanaServices.get().http.post(_constants.TIMELINE_URL, {
    method: 'POST',
    body: requestBody,
    version: '2023-10-31'
  });
  return decodeTimelineResponse(response);
};
const patchTimeline = async ({
  timelineId,
  timeline,
  version,
  savedSearch
}) => {
  let response = null;
  let requestBody = null;
  try {
    requestBody = JSON.stringify({
      timeline,
      timelineId,
      version
    });
  } catch (err) {
    return Promise.reject(new Error(`Failed to stringify query: ${JSON.stringify(err)}`));
  }
  try {
    if (timeline.savedSearchId && savedSearch) {
      const {
        savedSearch: savedSearchService
      } = _kibana.KibanaServices.get();
      await savedSearchService.save(savedSearch, {
        onTitleDuplicate: () => ({}),
        copyOnSave: false
      });
    }
  } catch (e) {
    return Promise.reject(new Error(`Failed to copy saved search: ${timeline.savedSearchId}`));
  }
  try {
    response = await _kibana.KibanaServices.get().http.patch(_constants.TIMELINE_URL, {
      method: 'PATCH',
      body: requestBody,
      version: '2023-10-31'
    });
  } catch (err) {
    // For Future developer
    // We are not rejecting our promise here because we had issue with our RXJS epic
    // the issue we were not able to pass the right object to it so we did manage the error in the success
    return Promise.resolve(decodeTimelineErrorResponse(err.body));
  }
  return decodeTimelineResponse(response);
};

/**
 * Creates a copy of the timeline with the given id. It will also apply changes to the original timeline
 * which are passed as `timeline` here.
 */
const copyTimeline = async ({
  timelineId,
  timeline,
  savedSearch
}) => {
  let response = null;
  let requestBody = null;
  let newSavedSearchId = null;
  try {
    if (timeline.savedSearchId && savedSearch) {
      const {
        savedSearch: savedSearchService
      } = _kibana.KibanaServices.get();
      const savedSearchCopy = {
        ...savedSearch
      };
      // delete the id and change the title to make sure we can copy the saved search
      delete savedSearchCopy.id;
      newSavedSearchId = await savedSearchService.save(savedSearchCopy, {
        onTitleDuplicate: () => ({}),
        copyOnSave: false
      });
    }
  } catch (e) {
    return Promise.reject(new Error(`Failed to copy saved search: ${timeline.savedSearchId}`));
  }
  try {
    requestBody = JSON.stringify({
      timeline: {
        ...timeline,
        savedSearchId: newSavedSearchId || timeline.savedSearchId
      },
      timelineIdToCopy: timelineId
    });
  } catch (err) {
    return Promise.reject(new Error(`Failed to stringify query: ${JSON.stringify(err)}`));
  }
  try {
    response = await _kibana.KibanaServices.get().http.post(_constants.TIMELINE_COPY_URL, {
      method: 'POST',
      body: requestBody,
      version: '1'
    });
  } catch (err) {
    // For Future developer
    // We are not rejecting our promise here because we had issue with our RXJS epic
    // the issue we were not able to pass the right object to it so we did manage the error in the success
    return Promise.resolve(decodeTimelineErrorResponse(err.body));
  }
  return decodeTimelineResponse(response);
};
exports.copyTimeline = copyTimeline;
const persistTimeline = async ({
  timelineId,
  timeline,
  version,
  savedSearch
}) => {
  try {
    if ((0, _lodash.isEmpty)(timelineId) && timeline.status === _timeline.TimelineStatusEnum.draft && timeline) {
      var _timeline$templateTim, _timeline$templateTim2, _draftTimeline$templa, _draftTimeline$templa2, _draftTimeline$versio;
      const temp = await cleanDraftTimeline({
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        timelineType: timeline.timelineType,
        templateTimelineId: (_timeline$templateTim = timeline.templateTimelineId) !== null && _timeline$templateTim !== void 0 ? _timeline$templateTim : undefined,
        templateTimelineVersion: (_timeline$templateTim2 = timeline.templateTimelineVersion) !== null && _timeline$templateTim2 !== void 0 ? _timeline$templateTim2 : undefined
      });
      const draftTimeline = decodeTimelineResponse(temp);
      const templateTimelineInfo = timeline.timelineType === _timeline.TimelineTypeEnum.template ? {
        templateTimelineId: (_draftTimeline$templa = draftTimeline.templateTimelineId) !== null && _draftTimeline$templa !== void 0 ? _draftTimeline$templa : timeline.templateTimelineId,
        templateTimelineVersion: (_draftTimeline$templa2 = draftTimeline.templateTimelineVersion) !== null && _draftTimeline$templa2 !== void 0 ? _draftTimeline$templa2 : timeline.templateTimelineVersion
      } : {};
      return patchTimeline({
        timelineId: draftTimeline.savedObjectId,
        timeline: {
          ...timeline,
          ...templateTimelineInfo
        },
        version: (_draftTimeline$versio = draftTimeline.version) !== null && _draftTimeline$versio !== void 0 ? _draftTimeline$versio : '',
        savedSearch
      });
    }
    if ((0, _lodash.isEmpty)(timelineId)) {
      return postTimeline({
        timeline
      });
    }
    return patchTimeline({
      timelineId: timelineId !== null && timelineId !== void 0 ? timelineId : '-1',
      timeline,
      version: version !== null && version !== void 0 ? version : '',
      savedSearch
    });
  } catch (err) {
    var _err$body;
    if (err.status_code === 403 || ((_err$body = err.body) === null || _err$body === void 0 ? void 0 : _err$body.status_code) === 403) {
      return Promise.resolve({
        statusCode: 403,
        message: err.message || err.body.message
      });
    }
    return Promise.resolve(err);
  }
};
exports.persistTimeline = persistTimeline;
const importTimelines = async ({
  fileToImport,
  signal
}) => {
  const formData = new FormData();
  formData.append('file', fileToImport);
  return _kibana.KibanaServices.get().http.fetch(`${_constants.TIMELINE_IMPORT_URL}`, {
    method: 'POST',
    headers: {
      'Content-Type': undefined
    },
    body: formData,
    signal,
    version: '2023-10-31'
  });
};
exports.importTimelines = importTimelines;
const exportSelectedTimeline = ({
  filename = `timelines_export.ndjson`,
  ids = [],
  signal
}) => {
  let requestBody;
  try {
    requestBody = ids.length > 0 ? JSON.stringify({
      ids
    }) : undefined;
  } catch (err) {
    return Promise.reject(new Error(`Failed to stringify query: ${JSON.stringify(err)}`));
  }
  return _kibana.KibanaServices.get().http.fetch(`${_constants.TIMELINE_EXPORT_URL}`, {
    method: 'POST',
    body: requestBody,
    query: {
      file_name: filename
    },
    signal,
    version: '2023-10-31'
  });
};
exports.exportSelectedTimeline = exportSelectedTimeline;
const getDraftTimeline = async ({
  timelineType
}) => {
  const response = await _kibana.KibanaServices.get().http.get(_constants.TIMELINE_DRAFT_URL, {
    query: {
      timelineType
    },
    version: '2023-10-31'
  });
  return decodeTimelineResponse(response);
};
exports.getDraftTimeline = getDraftTimeline;
const cleanDraftTimeline = async ({
  timelineType,
  templateTimelineId,
  templateTimelineVersion
}) => {
  let requestBody;
  const templateTimelineInfo = timelineType === _timeline.TimelineTypeEnum.template ? {
    templateTimelineId,
    templateTimelineVersion
  } : {};
  try {
    requestBody = JSON.stringify({
      timelineType,
      ...templateTimelineInfo
    });
  } catch (err) {
    return Promise.reject(new Error(`Failed to stringify query: ${JSON.stringify(err)}`));
  }
  const response = await _kibana.KibanaServices.get().http.post(_constants.TIMELINE_DRAFT_URL, {
    body: requestBody,
    version: '2023-10-31'
  });
  return decodeTimelineResponse(response);
};
exports.cleanDraftTimeline = cleanDraftTimeline;
const installPrepackedTimelines = async () => {
  const response = await _kibana.KibanaServices.get().http.post(_constants.TIMELINE_PREPACKAGED_URL, {
    version: '2023-10-31'
  });
  return decodePrepackedTimelineResponse(response);
};
exports.installPrepackedTimelines = installPrepackedTimelines;
const getTimeline = async id => {
  const response = await _kibana.KibanaServices.get().http.get(_constants.TIMELINE_URL, {
    query: {
      id
    },
    version: '2023-10-31'
  });
  return decodeSingleTimelineResponse(response);
};
exports.getTimeline = getTimeline;
const resolveTimeline = async id => {
  const response = await _kibana.KibanaServices.get().http.get(_constants.TIMELINE_RESOLVE_URL, {
    query: {
      id
    },
    version: '2023-10-31'
  });
  return decodeResolvedSingleTimelineResponse(response);
};
exports.resolveTimeline = resolveTimeline;
const getTimelineTemplate = async templateTimelineId => {
  const response = await _kibana.KibanaServices.get().http.get(_constants.TIMELINE_URL, {
    query: {
      template_timeline_id: templateTimelineId
    },
    version: '2023-10-31'
  });
  return decodeSingleTimelineResponse(response);
};
exports.getTimelineTemplate = getTimelineTemplate;
const getAllTimelines = async (query, abortSignal) => {
  const response = await _kibana.KibanaServices.get().http.fetch(_constants.TIMELINES_URL, {
    method: 'GET',
    query,
    signal: abortSignal,
    version: '2023-10-31'
  });
  return decodeGetTimelinesResponse(response);
};
exports.getAllTimelines = getAllTimelines;
const persistFavorite = async ({
  timelineId,
  templateTimelineId,
  templateTimelineVersion,
  timelineType
}) => {
  let requestBody;
  try {
    requestBody = JSON.stringify({
      timelineId,
      templateTimelineId,
      templateTimelineVersion,
      timelineType
    });
  } catch (err) {
    return Promise.reject(new Error(`Failed to stringify query: ${JSON.stringify(err)}`));
  }
  const response = await _kibana.KibanaServices.get().http.patch(_constants.TIMELINE_FAVORITE_URL, {
    method: 'PATCH',
    body: requestBody,
    version: '2023-10-31'
  });
  return decodeResponseFavoriteTimeline(response);
};
exports.persistFavorite = persistFavorite;
const deleteTimelinesByIds = async (savedObjectIds, searchIds) => {
  let requestBody;
  try {
    if (searchIds) {
      requestBody = JSON.stringify({
        savedObjectIds,
        searchIds
      });
    } else {
      requestBody = JSON.stringify({
        savedObjectIds
      });
    }
  } catch (err) {
    return Promise.reject(new Error(`Failed to stringify query: ${JSON.stringify(err)}`));
  }
  const response = await _kibana.KibanaServices.get().http.delete(_constants.TIMELINE_URL, {
    method: 'DELETE',
    body: requestBody,
    version: '2023-10-31'
  });
  return response;
};
exports.deleteTimelinesByIds = deleteTimelinesByIds;