"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDataTablesStorage = exports.migrateTriggerActionsVisibleColumnsAlertTable88xTo89 = exports.migrateLegacyTimelinesToSecurityDataTable = exports.migrateColumnWidthToInitialWidth = exports.migrateColumnLabelToDisplayAsText = exports.migrateAlertTableStateToTriggerActionsState = exports.getDataTablesInStorageByIds = exports.getAllDataTablesInStorage = exports.addTableInStorage = exports.addAssigneesSpecsToSecurityDataTableIfNeeded = exports.LOCAL_STORAGE_TABLE_KEY = void 0;
var _fp = require("lodash/fp");
var _securitysolutionDataTable = require("@kbn/securitysolution-data-table");
var _columns = require("../../../detections/configurations/security_solution_detections/columns");
var _constants = require("../../../../common/constants");
var _kibana = require("../../../common/lib/kibana");
var _migrates_risk_level_title = require("./migrates_risk_level_title");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const LOCAL_STORAGE_TABLE_KEY = exports.LOCAL_STORAGE_TABLE_KEY = 'securityDataTable';
const LOCAL_STORAGE_TIMELINE_KEY_LEGACY = 'timelines';
const EMPTY_TABLE = {};

/**
 * Migrates the values of the data table from the legacy timelines key to the securityDataTable key
 */
// eslint-disable-next-line @typescript-eslint/no-explicit-any
const migrateLegacyTimelinesToSecurityDataTable = legacyTimelineTables => {
  if (!legacyTimelineTables) {
    return EMPTY_TABLE;
  }
  return Object.keys(legacyTimelineTables).reduce((acc, timelineTableId) => {
    const timelineModel = legacyTimelineTables[timelineTableId];
    return {
      ...acc,
      [timelineTableId]: {
        defaultColumns: timelineModel.defaultColumns,
        dataViewId: timelineModel.dataViewId,
        excludedRowRendererIds: timelineModel.excludedRowRendererIds,
        filters: timelineModel.filters,
        indexNames: timelineModel.indexNames,
        loadingEventIds: timelineModel.loadingEventIds,
        isSelectAllChecked: timelineModel.isSelectAllChecked,
        itemsPerPage: timelineModel.itemsPerPage,
        itemsPerPageOptions: timelineModel.itemsPerPageOptions,
        showCheckboxes: timelineModel.showCheckboxes,
        selectAll: timelineModel.selectAll,
        id: timelineModel.id,
        title: timelineModel.title,
        initialized: timelineModel.initialized,
        updated: timelineModel.updated,
        sort: timelineModel.sort,
        selectedEventIds: timelineModel.selectedEventIds,
        deletedEventIds: timelineModel.deletedEventIds,
        totalCount: timelineModel.totalCount || 0,
        viewMode: _constants.VIEW_SELECTION.gridView,
        additionalFilters: {
          showBuildingBlockAlerts: false,
          showOnlyThreatIndicatorAlerts: false
        },
        ...(Array.isArray(timelineModel.columns) ? {
          columns: timelineModel.columns.map(migrateColumnWidthToInitialWidth).map(migrateColumnLabelToDisplayAsText)
        } : {})
      }
    };
  }, {});
};

/*
 *
 *  This migraiton only works for upgrading from
 *  8.7 -> 8.8. Please do not edit this migration for any
 *  future release.
 *
 *  If there is a migration that is required to be done for
 *  any future release. It should written as a saperate piece of code
 *  and should be called after this migration
 *
 * **/
exports.migrateLegacyTimelinesToSecurityDataTable = migrateLegacyTimelinesToSecurityDataTable;
const migrateAlertTableStateToTriggerActionsState = (storage, legacyDataTableState) => {
  const triggerActionsStateKey = {
    [_securitysolutionDataTable.TableId.alertsOnAlertsPage]: `detection-engine-alert-table-${_constants.ALERTS_TABLE_REGISTRY_CONFIG_IDS.ALERTS_PAGE}-gridView`,
    [_securitysolutionDataTable.TableId.alertsOnRuleDetailsPage]: `detection-engine-alert-table-${_constants.ALERTS_TABLE_REGISTRY_CONFIG_IDS.RULE_DETAILS}-gridView`
  };
  const triggersActionsState = Object.keys(legacyDataTableState).filter(tableKey => {
    return tableKey in triggerActionsStateKey && !storage.get(triggerActionsStateKey[tableKey]);
  }).map(tableKey => {
    const newKey = triggerActionsStateKey[tableKey];
    return {
      [newKey]: {
        columns: legacyDataTableState[tableKey].columns,
        sort: legacyDataTableState[tableKey].sort.map(sortCandidate => ({
          [sortCandidate.columnId]: {
            order: sortCandidate.sortDirection
          }
        })),
        visibleColumns: legacyDataTableState[tableKey].columns
      }
    };
  });
  triggersActionsState.forEach(stateObj => Object.keys(stateObj).forEach(key => {
    storage.set(key, stateObj[key]);
  }));
};

/*
 *
 * Used for migrating Alert Table from 8.8 => 8.9
 * */
exports.migrateAlertTableStateToTriggerActionsState = migrateAlertTableStateToTriggerActionsState;
const migrateTriggerActionsVisibleColumnsAlertTable88xTo89 = storage => {
  const localStorageKeys = [`detection-engine-alert-table-${_constants.ALERTS_TABLE_REGISTRY_CONFIG_IDS.ALERTS_PAGE}-gridView`, `detection-engine-alert-table-${_constants.ALERTS_TABLE_REGISTRY_CONFIG_IDS.RULE_DETAILS}-gridView`];
  localStorageKeys.forEach(key => {
    const alertTableData = storage.get(key);
    if (!alertTableData) {
      return;
    }
    if ('visibleColumns' in alertTableData) {
      const visibleColumns = alertTableData.visibleColumns;
      const v89CompliantFormat = visibleColumns.every(val => typeof val === 'string');
      if (v89CompliantFormat) {
        return;
      }
      const newVisibleColumns = visibleColumns.map(visibleCol => {
        if (typeof visibleCol === 'string') {
          // if column format is 8.9 compliant already
          return visibleCol;
        }
        // if column format is 8.8
        return visibleCol.id;
      }).filter(Boolean);
      storage.set(key, {
        ...alertTableData,
        visibleColumns: newVisibleColumns
      });
    }
  });
};

/**
 * Migrates the value of the column's `width` property to `initialWidth`
 * when `width` is valid, and `initialWidth` is invalid
 */
exports.migrateTriggerActionsVisibleColumnsAlertTable88xTo89 = migrateTriggerActionsVisibleColumnsAlertTable88xTo89;
const migrateColumnWidthToInitialWidth = column => ({
  ...column,
  ...(Number.isInteger(column.width) && !Number.isInteger(column.initialWidth) ? {
    initialWidth: column.width
  } : column.initialWidth ? {
    initialWidth: column.initialWidth
  } : {})
});

/**
 * Migrates the value of the column's `label` property to `displayAsText`
 * when `label` is valid, and `displayAsText` is `undefined`
 */
exports.migrateColumnWidthToInitialWidth = migrateColumnWidthToInitialWidth;
const migrateColumnLabelToDisplayAsText = column => ({
  ...column,
  ...(!(0, _fp.isEmpty)(column.label) && column.displayAsText == null ? {
    displayAsText: column.label
  } : column.displayAsText ? {
    displayAsText: column.displayAsText
  } : {})
});

/**
 * Adds "Assignees" column and makes it visible in alerts table
 */
exports.migrateColumnLabelToDisplayAsText = migrateColumnLabelToDisplayAsText;
const addAssigneesColumnToAlertsTable = storage => {
  const localStorageKeys = [`detection-engine-alert-table-${_constants.ALERTS_TABLE_REGISTRY_CONFIG_IDS.ALERTS_PAGE}-gridView`, `detection-engine-alert-table-${_constants.ALERTS_TABLE_REGISTRY_CONFIG_IDS.RULE_DETAILS}-gridView`];
  localStorageKeys.forEach(key => {
    const alertTableData = storage.get(key);
    if (!alertTableData) {
      return;
    }
    // Make "Assignees" field selected in the table
    if ('columns' in alertTableData) {
      let updatedAlertsTableState = false;
      const columns = alertTableData.columns;
      const hasAssigneesColumn = columns.findIndex(col => col.id === _columns.assigneesColumn.id) !== -1;
      if (!hasAssigneesColumn) {
        // Insert "Assignees" column at the index 1 to mimic behaviour of adding field to alerts table
        alertTableData.columns.splice(1, 0, _columns.assigneesColumn);
        updatedAlertsTableState = true;
      }
      // Make "Assignees" column visible in the table
      if ('visibleColumns' in alertTableData) {
        const visibleColumns = alertTableData.visibleColumns;
        const assigneesColumnExists = visibleColumns.findIndex(col => col === _columns.assigneesColumn.id) !== -1;
        if (!assigneesColumnExists) {
          alertTableData.visibleColumns.splice(1, 0, _columns.assigneesColumn.id);
          updatedAlertsTableState = true;
        }
      }
      if (updatedAlertsTableState) {
        storage.set(key, alertTableData);
      }
    }
  });
};

/**
 * Adds "Assignees" column specs to table data model
 */
const addAssigneesSpecsToSecurityDataTableIfNeeded = (storage, dataTableState) => {
  // Add "Assignees" column specs to the table data model
  let updatedTableModel = false;
  for (const [tableId, tableModel] of Object.entries(dataTableState)) {
    // Only add "Assignees" column specs to alerts tables
    if (_securitysolutionDataTable.tableEntity[tableId] !== _securitysolutionDataTable.TableEntityType.alert) {
      // eslint-disable-next-line no-continue
      continue;
    }

    // We added a new base column for "Assignees" in 8.12
    // In order to show correct custom header label after user upgrades to 8.12 we need to make sure the appropriate specs are in the table model.
    const columns = tableModel.columns;
    if (Array.isArray(columns)) {
      const hasAssigneesColumn = columns.findIndex(col => col.id === _columns.assigneesColumn.id) !== -1;
      if (!hasAssigneesColumn) {
        updatedTableModel = true;
        tableModel.columns.push(_columns.assigneesColumn);
      }
    }
    const defaultColumns = tableModel.defaultColumns;
    if (defaultColumns) {
      const hasAssigneesColumn = defaultColumns.findIndex(col => col.id === _columns.assigneesColumn.id) !== -1;
      if (!hasAssigneesColumn) {
        updatedTableModel = true;
        tableModel.defaultColumns.push(_columns.assigneesColumn);
      }
    }
  }
  if (updatedTableModel) {
    storage.set(LOCAL_STORAGE_TABLE_KEY, dataTableState);
    addAssigneesColumnToAlertsTable(storage);
  }
};
exports.addAssigneesSpecsToSecurityDataTableIfNeeded = addAssigneesSpecsToSecurityDataTableIfNeeded;
const getDataTablesInStorageByIds = (storage, tableIds) => {
  let allDataTables = storage.get(LOCAL_STORAGE_TABLE_KEY);
  const legacyTimelineTables = storage.get(LOCAL_STORAGE_TIMELINE_KEY_LEGACY);
  if (!allDataTables) {
    if (legacyTimelineTables) {
      allDataTables = migrateLegacyTimelinesToSecurityDataTable(legacyTimelineTables);
    } else {
      return EMPTY_TABLE;
    }
  }
  migrateAlertTableStateToTriggerActionsState(storage, allDataTables);
  migrateTriggerActionsVisibleColumnsAlertTable88xTo89(storage);
  addAssigneesSpecsToSecurityDataTableIfNeeded(storage, allDataTables);
  (0, _migrates_risk_level_title.migrateEntityRiskLevelColumnTitle)(storage, allDataTables);
  return tableIds.reduce((acc, tableId) => {
    const tableModel = allDataTables[tableId];
    if (!tableModel) {
      return {
        ...acc
      };
    }
    return {
      ...acc,
      [tableId]: {
        ...tableModel,
        ...(tableModel.sort != null && !Array.isArray(tableModel.sort) ? {
          sort: [tableModel.sort]
        } : {})
      }
    };
  }, {});
};
exports.getDataTablesInStorageByIds = getDataTablesInStorageByIds;
const getAllDataTablesInStorage = storage => {
  let allDataTables = storage.get(LOCAL_STORAGE_TABLE_KEY);
  const legacyTimelineTables = storage.get(LOCAL_STORAGE_TIMELINE_KEY_LEGACY);
  if (!allDataTables) {
    if (legacyTimelineTables) {
      allDataTables = migrateLegacyTimelinesToSecurityDataTable(legacyTimelineTables);
    } else {
      return EMPTY_TABLE;
    }
  }
  return allDataTables;
};
exports.getAllDataTablesInStorage = getAllDataTablesInStorage;
const addTableInStorage = (storage, id, table) => {
  const tableToStore = getSerializingTableToStore(table);
  const tables = getAllDataTablesInStorage(storage);
  storage.set(LOCAL_STORAGE_TABLE_KEY, {
    ...tables,
    [id]: tableToStore
  });
};
exports.addTableInStorage = addTableInStorage;
const getSerializingTableToStore = table => {
  // discard unneeded fields to make sure the object serialization works
  const {
    isLoading,
    loadingText,
    queryFields,
    unit,
    ...tableToStore
  } = table;
  return tableToStore;
};
const useDataTablesStorage = () => {
  const {
    storage
  } = (0, _kibana.useKibana)().services;
  const getAllDataTables = () => getAllDataTablesInStorage(storage);
  const getDataTablesById = id => {
    var _getDataTablesInStora;
    return (_getDataTablesInStora = getDataTablesInStorageByIds(storage, [id])[id]) !== null && _getDataTablesInStora !== void 0 ? _getDataTablesInStora : null;
  };
  const addDataTable = (id, table) => addTableInStorage(storage, id, table);
  return {
    getAllDataTables,
    getDataTablesById,
    addDataTable
  };
};
exports.useDataTablesStorage = useDataTablesStorage;