"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.migrateEndpointDataToSupportSpaces = exports.RESPONSE_ACTIONS_MIGRATION_REF_DATA_ID = exports.NOT_FOUND_VALUE = exports.ARTIFACTS_MIGRATION_REF_DATA_ID = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _common = require("@kbn/spaces-plugin/common");
var _pMap = _interopRequireDefault(require("p-map"));
var _common2 = require("@kbn/fleet-plugin/common");
var _lodash = require("lodash");
var _constants = require("../services/actions/constants");
var _artifacts = require("../../../common/endpoint/service/artifacts");
var _services = require("../services");
var _crowdstrike = require("../../../common/endpoint/service/response_actions/crowdstrike");
var _sentinel_one = require("../../../common/endpoint/service/response_actions/sentinel_one");
var _microsoft_defender = require("../../../common/endpoint/service/response_actions/microsoft_defender");
var _constants2 = require("../../../common/endpoint/service/response_actions/constants");
var _errors = require("../../../common/endpoint/errors");
var _create_es_search_iterable = require("../utils/create_es_search_iterable");
var _stringify = require("../utils/stringify");
var _reference_data = require("../lib/reference_data");
var _utils = require("../../../common/endpoint/service/artifacts/utils");
var _utils2 = require("../utils");
var _queue_processor = require("../utils/queue_processor");
var _constants3 = require("../../../common/endpoint/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable require-atomic-updates */

const LOGGER_KEY = 'migrateEndpointDataToSupportSpaces';
const ARTIFACTS_MIGRATION_REF_DATA_ID = exports.ARTIFACTS_MIGRATION_REF_DATA_ID = _reference_data.REF_DATA_KEYS.spaceAwarenessArtifactMigration;
const RESPONSE_ACTIONS_MIGRATION_REF_DATA_ID = exports.RESPONSE_ACTIONS_MIGRATION_REF_DATA_ID = _reference_data.REF_DATA_KEYS.spaceAwarenessResponseActionsMigration;
const NOT_FOUND_VALUE = exports.NOT_FOUND_VALUE = 'MIGRATION:NOT-FOUND';
const migrateEndpointDataToSupportSpaces = async endpointService => {
  const logger = endpointService.createLogger(LOGGER_KEY);
  if (!endpointService.experimentalFeatures.endpointManagementSpaceAwarenessEnabled) {
    logger.debug('Space awareness feature flag is disabled. Nothing to do.');
    return;
  }
  await Promise.all([migrateArtifactsToSpaceAware(endpointService), migrateResponseActionsToSpaceAware(endpointService)]);
};
exports.migrateEndpointDataToSupportSpaces = migrateEndpointDataToSupportSpaces;
const getMigrationState = async (refDataClient, id) => {
  return refDataClient.get(id);
};
const updateMigrationState = async (refDataClient, id, update) => {
  return refDataClient.update(id, update);
};
const migrateArtifactsToSpaceAware = async endpointService => {
  var _migrationState$metad;
  const logger = endpointService.createLogger(LOGGER_KEY, 'artifacts');
  const refDataClient = endpointService.getReferenceDataClient();
  const migrationState = await getMigrationState(refDataClient, ARTIFACTS_MIGRATION_REF_DATA_ID);

  // If migration has already been run and the version was 2 or higher -or- the status is pending (already running),
  // then we don't need to run it again
  if (migrationState.metadata.status === 'complete' && ((_migrationState$metad = migrationState.metadata.version) !== null && _migrationState$metad !== void 0 ? _migrationState$metad : 0) >= 2 || migrationState.metadata.status === 'pending') {
    logger.debug(`Migration (v2) for endpoint artifacts in support of spaces has a status of [${migrationState.metadata.status}], version [${migrationState.metadata.version}]. Nothing to do.`);
    return;
  }
  logger.debug(`starting migration (v2) of endpoint artifacts in support of spaces`);

  // If there are statuses about a prior run, then store a clone of it in the migration object for reference
  const priorRunData = migrationState.metadata.status === 'complete' ? (0, _lodash.clone)(migrationState.metadata) : undefined;

  // Migration version history:
  // V1: original migration of artifacts with release 9.1.0
  // v2: fixes for migration issue: https://github.com/elastic/kibana/issues/238711
  migrationState.metadata.version = 2;
  migrationState.metadata.status = 'pending';
  migrationState.metadata.started = new Date().toISOString();
  migrationState.metadata.finished = '';
  migrationState.metadata.data = undefined;
  await updateMigrationState(refDataClient, ARTIFACTS_MIGRATION_REF_DATA_ID, migrationState);
  const exceptionsClient = endpointService.getExceptionListsClient();
  const listIds = Object.values(_securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS).map(({
    id
  }) => id);
  logger.debug(`artifact list ids to process: ${listIds.join(', ')}`);
  const migrationStats = {
    totalItems: 0,
    itemsNeedingUpdates: 0,
    successUpdates: 0,
    failedUpdates: 0,
    artifacts: listIds.reduce((acc, listId) => {
      acc[listId] = {
        success: 0,
        failed: 0,
        errors: []
      };
      return acc;
    }, {}),
    priorRuns: []
  };
  migrationState.metadata.data = migrationStats;
  if (priorRunData) {
    var _priorRuns, _priorRunData$data;
    migrationStats.priorRuns.push(...((_priorRuns = (_priorRunData$data = priorRunData.data) === null || _priorRunData$data === void 0 ? void 0 : _priorRunData$data.priorRuns) !== null && _priorRuns !== void 0 ? _priorRuns : []));
    migrationStats.priorRuns.push({
      ...(0, _lodash.pick)(priorRunData, ['status', 'started', 'finished', 'version']),
      data: (0, _lodash.pick)(priorRunData.data, ['totalItems', 'itemsNeedingUpdates', 'successUpdates', 'failedUpdates', 'artifacts'])
    });
  }
  const updateProcessor = new _queue_processor.QueueProcessor({
    batchSize: 50,
    batchHandler: async ({
      data: artifactUpdates
    }) => {
      migrationStats.itemsNeedingUpdates += artifactUpdates.length;
      await (0, _pMap.default)(artifactUpdates, async ({
        listId,
        ...artifactUpdate
      }) => {
        try {
          const updatedArtifact = await exceptionsClient.updateExceptionListItem(artifactUpdate);
          if (updatedArtifact) {
            migrationStats.successUpdates++;
            migrationStats.artifacts[listId].success++;
          }
        } catch (err) {
          migrationStats.failedUpdates++;
          migrationStats.artifacts[listId].failed++;
          migrationStats.artifacts[listId].errors.push(`Update to [${listId}] item ID [${artifactUpdate.itemId}] failed with: ${err.message}`);
        }
      }, {
        stopOnError: false,
        concurrency: 10
      });
      await updateMigrationState(refDataClient, ARTIFACTS_MIGRATION_REF_DATA_ID, migrationState);
    }
  });
  await exceptionsClient.findExceptionListsItemPointInTimeFinder({
    listId: listIds,
    namespaceType: listIds.map(() => 'agnostic'),
    filter: listIds.map(
    // Find all artifacts that do NOT have a space owner id tag
    () => `NOT exception-list-agnostic.attributes.tags:(ownerSpaceId*)`),
    perPage: 1_000,
    sortField: undefined,
    sortOrder: undefined,
    maxSize: undefined,
    executeFunctionOnStream: ({
      page,
      total,
      data
    }) => {
      logger.debug(`Checking page [${page}] with [${data.length}] items out of a total of [${total}] artifact entries need updates`);
      if (migrationStats.totalItems < total) {
        migrationStats.totalItems = total;
      }
      for (const artifact of data) {
        if (!(0, _utils.hasArtifactOwnerSpaceId)(artifact)) {
          var _artifact$tags;
          const artifactUpdate = {
            _version: undefined,
            comments: artifact.comments,
            description: artifact.description,
            entries: artifact.entries,
            expireTime: artifact.expire_time,
            id: artifact.id,
            itemId: artifact.item_id,
            listId: artifact.list_id,
            meta: artifact.meta,
            name: artifact.name,
            namespaceType: artifact.namespace_type,
            osTypes: artifact.os_types,
            type: artifact.type,
            tags: [...((_artifact$tags = artifact.tags) !== null && _artifact$tags !== void 0 ? _artifact$tags : []), (0, _utils.buildSpaceOwnerIdTag)(_common.DEFAULT_SPACE_ID)]
          };

          // Ensure that Endpoint Exceptions all have the `global` tag if no assignment tag is currently assigned to the artifact
          if (artifact.list_id === _securitysolutionListConstants.ENDPOINT_LIST_ID && !(0, _utils.hasGlobalOrPerPolicyTag)(artifact)) {
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            artifactUpdate.tags.push(_artifacts.GLOBAL_ARTIFACT_TAG);
          }
          updateProcessor.addToQueue(artifactUpdate);
        }
      }
    }
  }).catch(_utils2.catchAndWrapError);
  await updateProcessor.complete();
  migrationState.metadata.status = 'complete';
  migrationState.metadata.finished = new Date().toISOString();
  await updateMigrationState(refDataClient, ARTIFACTS_MIGRATION_REF_DATA_ID, migrationState);
  logger.info(`migration of endpoint artifacts in support of spaces done.\n${JSON.stringify(migrationStats, null, 2)}`);
};
const migrateResponseActionsToSpaceAware = async endpointService => {
  const logger = endpointService.createLogger(LOGGER_KEY, 'responseActions');
  const refDataClient = endpointService.getReferenceDataClient();
  const migrationState = await getMigrationState(refDataClient, RESPONSE_ACTIONS_MIGRATION_REF_DATA_ID);
  if (migrationState.metadata.status !== 'not-started') {
    logger.debug(`Migration for endpoint response actions in support of spaces has a status of [${migrationState.metadata.status}]. Nothing to do.`);
    return;
  }
  logger.info(`starting migration of endpoint response actions in support of spaces`);
  const migrationStats = {
    totalItems: 0,
    itemsNeedingUpdates: 0,
    successUpdates: 0,
    failedUpdates: 0,
    warnings: [],
    errors: []
  };
  migrationState.metadata.status = 'pending';
  migrationState.metadata.started = new Date().toISOString();
  migrationState.metadata.data = migrationStats;
  await updateMigrationState(refDataClient, RESPONSE_ACTIONS_MIGRATION_REF_DATA_ID, migrationState);
  const indexInfo = await ensureResponseActionsIndexHasRequiredMappings(endpointService);
  if (indexInfo.error) {
    migrationStats.errors.push(indexInfo.error);
  }
  if (!indexInfo.successful) {
    migrationState.metadata.status = 'complete';
    migrationState.metadata.finished = new Date().toISOString();
    await updateMigrationState(refDataClient, RESPONSE_ACTIONS_MIGRATION_REF_DATA_ID, migrationState);
    return;
  }
  if (!indexInfo.indexExists) {
    const exitMessage = `Response actions index [${_constants3.ENDPOINT_ACTIONS_INDEX}] does not exist. Nothing to migrate`;
    logger.debug(exitMessage);
    migrationState.metadata.status = 'complete';
    migrationState.metadata.finished = new Date().toISOString();
    migrationStats.warnings.push(exitMessage);
    await updateMigrationState(refDataClient, RESPONSE_ACTIONS_MIGRATION_REF_DATA_ID, migrationState);
    return;
  }
  const policyInfoBuilder = new AgentPolicyInfoBuilder(endpointService, logger);
  const esClient = endpointService.getInternalEsClient();
  const updateProcessor = new _queue_processor.QueueProcessor({
    batchSize: 50,
    logger,
    batchHandler: async ({
      data: actionUpdates
    }) => {
      migrationStats.itemsNeedingUpdates += actionUpdates.length;
      const docIdToResponseActionIdMap = {};
      const bulkOperations = [];
      for (const actionUpdateInfo of actionUpdates) {
        docIdToResponseActionIdMap[actionUpdateInfo.docId] = actionUpdateInfo.actionId;
        bulkOperations.push({
          update: {
            _index: actionUpdateInfo.index,
            _id: actionUpdateInfo.docId
          }
        }, {
          doc: actionUpdateInfo.update
        });
      }
      logger.debug(`Calling esClient.bulk() with [${actionUpdates.length}] updates`);
      try {
        const bulkUpdateResponse = await esClient.bulk({
          operations: bulkOperations
        });
        if (!bulkUpdateResponse.errors) {
          migrationStats.successUpdates += actionUpdates.length;
        } else {
          for (const operationResult of bulkUpdateResponse.items) {
            const updateResponse = operationResult.update;
            if (updateResponse !== null && updateResponse !== void 0 && updateResponse.error) {
              var _updateResponse$_id;
              migrationStats.failedUpdates++;
              migrationStats.errors.push(`Update to response action [${docIdToResponseActionIdMap[(_updateResponse$_id = updateResponse._id) !== null && _updateResponse$_id !== void 0 ? _updateResponse$_id : '']}] _id [${updateResponse._id}] failed: ${updateResponse.error.reason}`);
            } else {
              migrationStats.successUpdates++;
            }
          }
        }
      } catch (err) {
        logger.error(`ES Bulk update failed with: ${(0, _stringify.stringify)(err)}`);
        migrationStats.failedUpdates += actionUpdates.length;
        migrationStats.errors.push(`Bulk update of action ids [${Object.values(docIdToResponseActionIdMap).join(', ')}] failed with: ${err.message}`);
      }

      // Write stats so we can see intermediate stats if required
      await updateMigrationState(refDataClient, RESPONSE_ACTIONS_MIGRATION_REF_DATA_ID, migrationState);
    }
  });
  const responseActionsFetcher = (0, _create_es_search_iterable.createEsSearchIterable)({
    esClient,
    searchRequest: {
      index: _constants3.ENDPOINT_ACTIONS_INDEX,
      sort: '@timestamp',
      size: 50,
      query: {
        bool: {
          must_not: {
            exists: {
              field: 'originSpaceId'
            }
          }
        }
      }
    }
  });
  const updateBuilderPromises = [];
  for await (const actionRequestSearchResults of responseActionsFetcher) {
    var _value;
    const actionRequests = actionRequestSearchResults.hits.hits;
    const totalItems = (_value = actionRequestSearchResults.hits.total.value) !== null && _value !== void 0 ? _value : 0;
    if (migrationStats.totalItems !== totalItems) {
      migrationStats.totalItems = totalItems;
    }
    for (const actionHit of actionRequests) {
      const action = actionHit._source;
      if (action) {
        updateBuilderPromises.push(
        // We don't `await` here. These can just run in the background so that migration runs a little faster
        policyInfoBuilder.buildPolicyUpdate(action).then(updateContent => {
          if (updateContent.warnings.length > 0) {
            migrationStats.warnings.push(...updateContent.warnings);
          }
          updateProcessor.addToQueue({
            actionId: action.EndpointActions.action_id,
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            docId: actionHit._id,
            index: actionHit._index,
            update: updateContent.policyUpdate
          });
        }));
      }
    }
  }
  await Promise.allSettled(updateBuilderPromises);
  await updateProcessor.complete();
  migrationState.metadata.status = 'complete';
  migrationState.metadata.finished = new Date().toISOString();
  await updateMigrationState(refDataClient, RESPONSE_ACTIONS_MIGRATION_REF_DATA_ID, migrationState);
  logger.info(`migration of endpoint response actions in support of spaces done.\n${JSON.stringify(migrationStats, null, 2)}`);
};
class AgentPolicyInfoBuilder {
  constructor(endpointService, logger) {
    // Cache of previously processed Agent IDs
    (0, _defineProperty2.default)(this, "agentIdCache", new Map());
    (0, _defineProperty2.default)(this, "agentIdToAgentPolicyIdCache", new Map());
    (0, _defineProperty2.default)(this, "agentPolicyIdToIntegrationPolicyId", new Map());
    (0, _defineProperty2.default)(this, "externalEdrAgentIdToElasticAgentIdMapCache", new Map());
    this.endpointService = endpointService;
    this.logger = logger;
  }
  async buildPolicyUpdate(actionRequest) {
    // TODO: trigger retrieval of agent records and package polices for those agents in bulk??

    const agentsPolicyInfo = (Array.isArray(actionRequest.agent.id) ? actionRequest.agent.id : [actionRequest.agent.id]).map(agentId => {
      let agentPolicyInfoPromise = this.agentIdCache.get(agentId);
      if (!agentPolicyInfoPromise) {
        this.logger.debug(`Agent ID [${agentId}] policy info content not yet created. Building it now`);
        agentPolicyInfoPromise = this.fetchAgentPolicyInfo(agentId, actionRequest);
        this.agentIdCache.set(agentId, agentPolicyInfoPromise);
        return agentPolicyInfoPromise;
      } else {
        this.logger.debug(`Found cached policy info for agent ID [${agentId}]. No need to build it.`);
      }
      return agentPolicyInfoPromise;
    });
    const response = {
      warnings: [],
      policyUpdate: {
        originSpaceId: _common.DEFAULT_SPACE_ID,
        agent: {
          policy: []
        }
      }
    };
    await Promise.all(agentsPolicyInfo).then(agentsInfo => {
      for (const agentPolicyInfo of agentsInfo) {
        if (agentPolicyInfo.warnings.length > 0) {
          response.warnings.push(...agentPolicyInfo.warnings);
        }
        response.policyUpdate.agent.policy.push(agentPolicyInfo.agentInfo);
        if (agentPolicyInfo.agentInfo.integrationPolicyId === NOT_FOUND_VALUE) {
          response.policyUpdate.tags = [...(actionRequest.tags || []), _constants.ALLOWED_ACTION_REQUEST_TAGS.integrationPolicyDeleted];
        }
      }
    });
    this.logger.debug(() => `Action [${actionRequest.EndpointActions.action_id}] update: ${JSON.stringify(response)}`);
    return response;
  }
  async fetchAgentPolicyInfo(_agentId, actionRequest) {
    let agentId = _agentId;
    if (actionRequest.EndpointActions.input_type !== 'endpoint') {
      const externalEdrAgentId = await this.fetchElasticAgentIdFor3rdPartyEdr(_agentId, actionRequest);
      if (!externalEdrAgentId.found) {
        return {
          warnings: [externalEdrAgentId.warning],
          agentInfo: {
            agentId: _agentId,
            elasticAgentId: NOT_FOUND_VALUE,
            agentPolicyId: NOT_FOUND_VALUE,
            integrationPolicyId: NOT_FOUND_VALUE
          }
        };
      }
      agentId = externalEdrAgentId.agentId;
    }
    const agentPolicyId = await this.fetchAgentPolicyIdForAgent(agentId);

    // If the agent policy is not found, then no need to go any further because we will not be able
    // to determine what integration policy the agent was/is running with.
    if (!agentPolicyId.found) {
      return {
        warnings: [agentPolicyId.warning],
        agentInfo: {
          agentId: _agentId,
          elasticAgentId: agentId,
          agentPolicyId: NOT_FOUND_VALUE,
          integrationPolicyId: NOT_FOUND_VALUE
        }
      };
    }
    const integrationPolicyId = await this.fetchAgentPolicyIntegrationPolicyId(agentPolicyId.policyId, actionRequest.EndpointActions.input_type);
    return {
      warnings: integrationPolicyId.found ? [] : [integrationPolicyId.warning],
      agentInfo: {
        agentId: _agentId,
        elasticAgentId: agentId,
        agentPolicyId: agentPolicyId.policyId,
        integrationPolicyId: integrationPolicyId.policyId
      }
    };
  }
  async fetchAgentPolicyIdForAgent(agentId) {
    let agentPolicyIdPromise = this.agentIdToAgentPolicyIdCache.get(agentId);
    if (!agentPolicyIdPromise) {
      const fleetServices = this.endpointService.getInternalFleetServices(undefined, true);
      agentPolicyIdPromise = fleetServices.agent.getAgent(agentId).then(agent => {
        if (!agent.policy_id) {
          throw new _errors.EndpointError(`Agent id [${agentId}] was found, but it does not contain a 'policy_id'`);
        }
        return {
          found: true,
          warning: '',
          policyId: agent.policy_id
        };
      }).catch(err => {
        return {
          found: false,
          warning: `Unable to retrieve agent id [${agentId}]: ${err.message}`,
          policyId: NOT_FOUND_VALUE
        };
      });
      this.agentIdToAgentPolicyIdCache.set(agentId, agentPolicyIdPromise);
      return agentPolicyIdPromise;
    }
    return agentPolicyIdPromise;
  }
  async fetchAgentPolicyIntegrationPolicyId(agentPolicyId, agentType) {
    // An agent policy - especially one for agentless integration (3rd party EDRs) - could include
    // multiple integration for the supported agent types, so we keep a cache for each agent type
    const cacheKey = `${agentPolicyId}#${agentType}`;
    let integrationPolicyIdPromise = this.agentPolicyIdToIntegrationPolicyId.get(cacheKey);
    if (!integrationPolicyIdPromise) {
      const fleetServices = this.endpointService.getInternalFleetServices(undefined, true);
      integrationPolicyIdPromise = fleetServices.packagePolicy.list(fleetServices.getSoClient(), {
        kuery: `${_common2.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name: (${_constants2.RESPONSE_ACTIONS_SUPPORTED_INTEGRATION_TYPES[agentType].join(' OR ')}) AND (${_common2.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.policy_ids:"${agentPolicyId}")`,
        perPage: 1
      }).then(response => {
        const integrationPolicy = response.items[0];
        if (!integrationPolicy) {
          throw new _errors.EndpointError(`Unable to find [${agentType}] integration policies for agent policy id [${agentPolicyId}]`);
        }
        return {
          found: true,
          warning: '',
          policyId: integrationPolicy.id
        };
      }).catch(err => {
        return {
          found: false,
          warning: err instanceof _errors.EndpointError ? err.message : `Failed to find [${agentType}] integration polices for agent policy [${agentPolicyId}]: ${err.message}`,
          policyId: NOT_FOUND_VALUE
        };
      });
      this.agentPolicyIdToIntegrationPolicyId.set(cacheKey, integrationPolicyIdPromise);
    }
    return integrationPolicyIdPromise;
  }
  async fetchElasticAgentIdFor3rdPartyEdr(externalEdrAgentId, actionRequest) {
    const agentType = actionRequest.EndpointActions.input_type;
    const elasticAgentIdPromise = this.externalEdrAgentIdToElasticAgentIdMapCache.get(externalEdrAgentId);
    if (elasticAgentIdPromise) {
      this.logger.debug(`Returning cached result [${agentType}] agent id [${externalEdrAgentId}]`);
      return elasticAgentIdPromise;
    }
    this.externalEdrAgentIdToElasticAgentIdMapCache.set(externalEdrAgentId, new Promise(async (resolve, reject) => {
      try {
        var _esSearchResult$hits$;
        const esClient = this.endpointService.getInternalEsClient();
        this.logger.debug(`Retrieving Elastic agent id for [${agentType}] agent [${externalEdrAgentId}]`);
        const esSearchRequest = {
          index: '',
          // value by switch statement below
          // query: {} set by switch statement below
          _source: ['agent'],
          sort: [{
            'event.created': 'desc'
          }],
          ignore_unavailable: true,
          allow_no_indices: true,
          size: 1
        };
        switch (agentType) {
          case 'microsoft_defender_endpoint':
            esSearchRequest.index = _microsoft_defender.MICROSOFT_DEFENDER_ENDPOINT_LOG_INDEX_PATTERN;
            esSearchRequest.query = {
              bool: {
                filter: [{
                  term: {
                    'cloud.instance.id': externalEdrAgentId
                  }
                }]
              }
            };
            break;
          case 'sentinel_one':
            esSearchRequest.index = _sentinel_one.SENTINEL_ONE_AGENT_INDEX_PATTERN;
            esSearchRequest.query = {
              bool: {
                filter: [{
                  term: {
                    'sentinel_one.agent.agent.id': externalEdrAgentId
                  }
                }]
              }
            };
            break;
          case 'crowdstrike':
            esSearchRequest.index = _crowdstrike.CROWDSTRIKE_HOST_INDEX_PATTERN;
            esSearchRequest.query = {
              bool: {
                filter: [{
                  term: {
                    'device.id': externalEdrAgentId
                  }
                }]
              }
            };
            break;
          default:
            resolve({
              found: false,
              warning: `Response action [${actionRequest.EndpointActions.action_id}] has an unsupported agent type [${agentType}].]`,
              agentId: ''
            });
            return;
        }
        this.logger.debug(() => `Searching [${agentType}] data to identify elastic agent used to ingest data for external agent id [${externalEdrAgentId}]:${(0, _stringify.stringify)(esSearchRequest)}`);
        const esSearchResult = await esClient.search(esSearchRequest).catch(_utils2.catchAndWrapError);
        const hitSource = (_esSearchResult$hits$ = esSearchResult.hits.hits[0]) === null || _esSearchResult$hits$ === void 0 ? void 0 : _esSearchResult$hits$._source;
        if (hitSource && hitSource.agent.id) {
          resolve({
            found: true,
            warning: '',
            agentId: hitSource.agent.id
          });
          return;
        }
        resolve({
          found: false,
          warning: `Unable to determine elastic agent id used to ingest data for [${agentType}] external agent id [${externalEdrAgentId}]`,
          agentId: ''
        });
      } catch (error) {
        reject(error);
      }
    }));

    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    return this.externalEdrAgentIdToElasticAgentIdMapCache.get(externalEdrAgentId);
  }
}
const ensureResponseActionsIndexHasRequiredMappings = async endpointService => {
  const logger = endpointService.createLogger(LOGGER_KEY, 'ensureResponseActionsIndexHasRequiredMappings');
  logger.debug(`Checking index [${_constants3.ENDPOINT_ACTIONS_INDEX}] mappings`);
  const response = {
    successful: true,
    indexExists: true,
    error: undefined
  };
  const esClient = endpointService.getInternalEsClient();
  const fleetServices = endpointService.getInternalFleetServices();
  const [indexExists, installedEndpointPackages] = await Promise.all([esClient.indices.exists({
    index: _constants3.ENDPOINT_ACTIONS_INDEX
  }).catch(_utils2.catchAndWrapError), fleetServices.packages.getInstalledPackages({
    perPage: 100,
    sortOrder: 'desc',
    nameQuery: 'endpoint'
  })]);
  response.indexExists = indexExists;
  logger.debug(() => `Does DS index already exists: ${indexExists}\nCurrently installed endpoint package:${(0, _stringify.stringify)(installedEndpointPackages)}`);

  // If the index does not exist and Endpoint package is not installed, then this must be an env.
  // where the use of security and/or Fleet is not being utilized.
  if (!indexExists && installedEndpointPackages.total === 0) {
    logger.debug(`Index [${_constants3.ENDPOINT_ACTIONS_INDEX}] does not yet exist and no endpoint package installed. Nothing to do.`);
    return response;
  }

  // if we got this far, then endpoint package is installed. Ensure index exists and add mappings
  await (0, _services.ensureActionRequestsIndexIsConfigured)(endpointService).catch(error => {
    response.error = `Attempt to add new mappings to [${_constants3.ENDPOINT_ACTIONS_INDEX}] index failed with:${(0, _stringify.stringify)(error)}`;
    response.successful = false;
    return response;
  });
  response.indexExists = true;
  logger.debug(`New mappings to index [${_constants3.ENDPOINT_ACTIONS_INDEX}] have been added successfully.`);
  return response;
};