"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.reviewRuleUpgradeHandler = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _utils = require("../../../routes/utils");
var _calculate_rule_diff = require("../../logic/diff/calculate_rule_diff");
var _prebuilt_rule_assets_client = require("../../logic/rule_assets/prebuilt_rule_assets_client");
var _prebuilt_rule_objects_client = require("../../logic/rule_objects/prebuilt_rule_objects_client");
var _zip_rule_versions = require("../../logic/rule_versions/zip_rule_versions");
var _calculate_rule_upgrade_info = require("./calculate_rule_upgrade_info");
var _utils2 = require("../../logic/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_SORT = {
  field: 'name',
  order: 'asc'
};
const reviewRuleUpgradeHandler = async (context, request, response) => {
  var _request$body;
  const siemResponse = (0, _utils.buildSiemResponse)(response);
  const {
    page = 1,
    per_page: perPage = 20,
    sort = DEFAULT_SORT,
    filter
  } = (_request$body = request.body) !== null && _request$body !== void 0 ? _request$body : {};
  try {
    const ctx = await context.resolve(['core', 'alerting', 'securitySolution']);
    const soClient = ctx.core.savedObjects.client;
    const rulesClient = await ctx.alerting.getRulesClient();
    const ruleAssetsClient = (0, _prebuilt_rule_assets_client.createPrebuiltRuleAssetsClient)(soClient);
    const ruleObjectsClient = (0, _prebuilt_rule_objects_client.createPrebuiltRuleObjectsClient)(rulesClient);
    const mlAuthz = ctx.securitySolution.getMlAuthz();
    const {
      diffResults,
      totalUpgradeableRules
    } = await calculateUpgradeableRulesDiff({
      ruleAssetsClient,
      ruleObjectsClient,
      mlAuthz,
      page,
      perPage,
      sort,
      filter
    });
    const body = {
      stats: {
        num_rules_to_upgrade_total: 0,
        num_rules_with_conflicts: 0,
        num_rules_with_non_solvable_conflicts: 0,
        tags: []
      },
      rules: (0, _calculate_rule_upgrade_info.calculateRuleUpgradeInfo)(diffResults),
      page,
      per_page: perPage,
      total: totalUpgradeableRules
    };
    return response.ok({
      body
    });
  } catch (err) {
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    return siemResponse.error({
      body: error.message,
      statusCode: error.statusCode
    });
  }
};
exports.reviewRuleUpgradeHandler = reviewRuleUpgradeHandler;
async function calculateUpgradeableRulesDiff({
  ruleAssetsClient,
  ruleObjectsClient,
  mlAuthz,
  page,
  perPage,
  sort,
  filter
}) {
  const allLatestVersions = await ruleAssetsClient.fetchLatestVersions();
  const latestVersionsMap = new Map(allLatestVersions.map(version => [version.rule_id, version]));
  const currentRuleVersions = filter !== null && filter !== void 0 && filter.rule_ids ? await ruleObjectsClient.fetchInstalledRuleVersionsByIds({
    ruleIds: filter.rule_ids,
    sortField: sort.field,
    sortOrder: sort.order
  }) : await ruleObjectsClient.fetchInstalledRuleVersions({
    filter,
    sortField: sort.field,
    sortOrder: sort.order
  });
  const upgradeableRules = await (0, _utils2.getPossibleUpgrades)(currentRuleVersions, latestVersionsMap, mlAuthz);
  const totalUpgradeableRules = upgradeableRules.length;
  const pagedRuleIds = upgradeableRules.slice((page - 1) * perPage, page * perPage).map(rule => rule.rule_id);
  const currentRules = await ruleObjectsClient.fetchInstalledRulesByIds({
    ruleIds: pagedRuleIds,
    sortField: sort.field,
    sortOrder: sort.order
  });
  const latestRules = await ruleAssetsClient.fetchAssetsByVersion(currentRules.map(({
    rule_id: ruleId
  }) => latestVersionsMap.get(ruleId)));
  const baseRules = await ruleAssetsClient.fetchAssetsByVersion(currentRules);
  const ruleVersionsMap = (0, _zip_rule_versions.zipRuleVersions)(currentRules, baseRules, latestRules);

  // Calculate the diff between current, base, and target versions
  // Iterate through the current rules array to keep the order of the results
  const diffResults = currentRules.map(current => {
    var _ruleVersionsMap$get, _ruleVersionsMap$get2;
    const base = (_ruleVersionsMap$get = ruleVersionsMap.get(current.rule_id)) === null || _ruleVersionsMap$get === void 0 ? void 0 : _ruleVersionsMap$get.base;
    const target = (_ruleVersionsMap$get2 = ruleVersionsMap.get(current.rule_id)) === null || _ruleVersionsMap$get2 === void 0 ? void 0 : _ruleVersionsMap$get2.target;
    return (0, _calculate_rule_diff.calculateRuleDiff)({
      current,
      base,
      target
    });
  });
  return {
    diffResults,
    totalUpgradeableRules
  };
}