"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.startEntityStoreFieldRetentionEnrichTask = exports.runEntityStoreFieldRetentionEnrichTask = exports.removeEntityStoreFieldRetentionEnrichTask = exports.registerEntityStoreFieldRetentionEnrichTask = exports.getEntityStoreFieldRetentionEnrichTaskState = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _server = require("@kbn/core/server");
var _managementSettingsIds = require("@kbn/management-settings-ids");
var _utils = require("../../../../../../common/entity_analytics/utils");
var _entity_store = require("../../../../../../common/api/entity_analytics/entity_store");
var _state = require("./state");
var _constants = require("./constants");
var _elasticsearch_assets = require("../../elasticsearch_assets");
var _utils2 = require("../../utils");
var _events = require("../../../../telemetry/event_based/events");
var _constants2 = require("../../entity_definitions/constants");
var _utils3 = require("../utils");
var _api_key = require("../../auth/api_key");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getTaskName = () => _constants.TYPE;
const getTaskId = namespace => `${_constants.TYPE}:${namespace}:${_constants.VERSION}`;
const registerEntityStoreFieldRetentionEnrichTask = ({
  getStartServices,
  logger,
  telemetry,
  taskManager
}) => {
  if (!taskManager) {
    logger.info('[Entity Store]  Task Manager is unavailable; skipping entity store enrich policy registration.');
    return;
  }
  const executeEnrichPolicy = async (namespace, entityType) => {
    const [coreStart, _] = await getStartServices();
    const esClient = coreStart.elasticsearch.client.asInternalUser;
    return (0, _elasticsearch_assets.executeFieldRetentionEnrichPolicy)({
      entityType,
      version: _constants2.VERSIONS_BY_ENTITY_TYPE[entityType],
      options: {
        namespace
      },
      esClient,
      logger
    });
  };
  const getStoreSize = async index => {
    const [coreStart] = await getStartServices();
    const esClient = coreStart.elasticsearch.client.asInternalUser;
    const {
      count
    } = await esClient.count({
      index
    });
    return count;
  };
  const getEnabledEntityTypesForNamespace = async namespace => {
    const [core, {
      security,
      encryptedSavedObjects
    }] = await getStartServices();
    const apiKeyManager = (0, _api_key.getApiKeyManager)({
      core,
      logger,
      security,
      encryptedSavedObjects,
      namespace
    });
    const apiKey = await apiKeyManager.getApiKey();
    if (!apiKey) {
      logger.info(`[Entity Store] No API key found, returning all entity types as enabled in ${namespace} namespace`);
      return (0, _utils.getEnabledEntityTypes)(true);
    }
    const {
      soClient
    } = await apiKeyManager.getClientFromApiKey(apiKey);
    const uiSettingsClient = core.uiSettings.asScopedToClient(soClient);
    const genericEntityStoreEnabled = await uiSettingsClient.get(_managementSettingsIds.SECURITY_SOLUTION_ENABLE_ASSET_INVENTORY_SETTING);
    return (0, _utils.getEnabledEntityTypes)(genericEntityStoreEnabled);
  };
  taskManager.registerTaskDefinitions({
    [getTaskName()]: {
      title: 'Entity Analytics Entity Store - Execute Enrich Policy Task',
      timeout: _constants.TIMEOUT,
      stateSchemaByVersion: _state.stateSchemaByVersion,
      createTaskRunner: createEntityStoreFieldRetentionEnrichTaskRunnerFactory({
        logger,
        telemetry,
        getStoreSize,
        executeEnrichPolicy,
        getEnabledEntityTypesForNamespace
      })
    }
  });
};
exports.registerEntityStoreFieldRetentionEnrichTask = registerEntityStoreFieldRetentionEnrichTask;
const startEntityStoreFieldRetentionEnrichTask = async ({
  logger,
  namespace,
  taskManager,
  interval
}) => {
  const taskId = getTaskId(namespace);
  const log = (0, _utils3.entityStoreTaskLogFactory)(logger, taskId);
  log('attempting to schedule');
  try {
    await taskManager.ensureScheduled({
      id: taskId,
      taskType: getTaskName(),
      scope: _constants.SCOPE,
      schedule: {
        interval
      },
      state: {
        ..._state.defaultState,
        namespace
      },
      params: {
        version: _constants.VERSION
      }
    });
  } catch (e) {
    logger.warn(`[Entity Store]  [task ${taskId}]: error scheduling task, received ${e.message}`);
    throw e;
  }
};
exports.startEntityStoreFieldRetentionEnrichTask = startEntityStoreFieldRetentionEnrichTask;
const removeEntityStoreFieldRetentionEnrichTask = async ({
  logger,
  namespace,
  taskManager
}) => {
  try {
    await taskManager.remove(getTaskId(namespace));
    logger.info(`[Entity Store]  Removed entity store enrich policy task for namespace ${namespace}`);
  } catch (err) {
    if (!_server.SavedObjectsErrorHelpers.isNotFoundError(err)) {
      logger.error(`[Entity Store]  Failed to remove  entity store enrich policy task: ${err.message}`);
      throw err;
    }
  }
};
exports.removeEntityStoreFieldRetentionEnrichTask = removeEntityStoreFieldRetentionEnrichTask;
const runEntityStoreFieldRetentionEnrichTask = async ({
  executeEnrichPolicy,
  getStoreSize,
  isCancelled,
  logger,
  taskInstance,
  telemetry,
  getEnabledEntityTypesForNamespace
}) => {
  const state = taskInstance.state;
  const taskId = taskInstance.id;
  const log = (0, _utils3.entityStoreTaskLogFactory)(logger, taskId);
  const debugLog = (0, _utils3.entityStoreTaskDebugLogFactory)(logger, taskId);
  try {
    var _taskInstance$schedul;
    const taskStartTime = (0, _moment.default)().utc().toISOString();
    log('running task');
    const updatedState = {
      lastExecutionTimestamp: taskStartTime,
      namespace: state.namespace,
      runs: state.runs + 1
    };
    if (taskId !== getTaskId(state.namespace)) {
      log('outdated task; exiting');
      return {
        state: updatedState
      };
    }
    const entityTypes = await getEnabledEntityTypesForNamespace(state.namespace);
    for (const entityType of entityTypes) {
      const start = Date.now();
      debugLog(`executing field retention enrich policy for ${entityType}`);
      try {
        const {
          executed
        } = await executeEnrichPolicy(state.namespace, entityType);
        if (!executed) {
          debugLog(`Field retention enrich policy for ${entityType} does not exist`);
        } else {
          log(`Executed field retention enrich policy for ${entityType} in ${Date.now() - start}ms`);
        }
      } catch (e) {
        log(`error executing field retention enrich policy for ${entityType}: ${e.message}`);
      }
    }
    const taskCompletionTime = (0, _moment.default)().utc().toISOString();
    const taskDurationInSeconds = (0, _moment.default)(taskCompletionTime).diff((0, _moment.default)(taskStartTime), 'seconds');
    log(`task run completed in ${taskDurationInSeconds} seconds`);
    telemetry.reportEvent(_events.FIELD_RETENTION_ENRICH_POLICY_EXECUTION_EVENT.eventType, {
      duration: taskDurationInSeconds,
      interval: (_taskInstance$schedul = taskInstance.schedule) === null || _taskInstance$schedul === void 0 ? void 0 : _taskInstance$schedul.interval
    });

    // Track entity store usage
    const indices = entityTypes.map(entityType => (0, _utils2.getEntitiesIndexName)(entityType, state.namespace));
    const storeSize = await getStoreSize(indices);
    telemetry.reportEvent(_events.ENTITY_STORE_USAGE_EVENT.eventType, {
      storeSize
    });
    return {
      state: updatedState
    };
  } catch (e) {
    logger.error(`[Entity Store] [task ${taskId}]: error running task, received ${e.message}`);
    throw e;
  }
};
exports.runEntityStoreFieldRetentionEnrichTask = runEntityStoreFieldRetentionEnrichTask;
const createEntityStoreFieldRetentionEnrichTaskRunnerFactory = ({
  logger,
  telemetry,
  executeEnrichPolicy,
  getStoreSize,
  getEnabledEntityTypesForNamespace
}) => ({
  taskInstance
}) => {
  let cancelled = false;
  const isCancelled = () => cancelled;
  return {
    run: async () => runEntityStoreFieldRetentionEnrichTask({
      executeEnrichPolicy,
      getStoreSize,
      isCancelled,
      logger,
      taskInstance,
      telemetry,
      getEnabledEntityTypesForNamespace
    }),
    cancel: async () => {
      cancelled = true;
    }
  };
};
const getEntityStoreFieldRetentionEnrichTaskState = async ({
  namespace,
  taskManager
}) => {
  const taskId = getTaskId(namespace);
  try {
    const taskState = await taskManager.get(taskId);
    return {
      id: taskState.id,
      resource: _entity_store.EngineComponentResourceEnum.task,
      installed: true,
      enabled: taskState.enabled,
      status: taskState.status,
      retryAttempts: taskState.attempts,
      nextRun: taskState.runAt,
      lastRun: taskState.state.lastExecutionTimestamp,
      runs: taskState.state.runs
    };
  } catch (e) {
    if (_server.SavedObjectsErrorHelpers.isNotFoundError(e)) {
      return {
        id: taskId,
        installed: false,
        resource: _entity_store.EngineComponentResourceEnum.task
      };
    }
    throw e;
  }
};
exports.getEntityStoreFieldRetentionEnrichTaskState = getEntityStoreFieldRetentionEnrichTaskState;