"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RuleMigrationTelemetryClient = void 0;
var _siem_migrations = require("../../../telemetry/event_based/events/siem_migrations");
var _siem_migrations_telemetry_client = require("../../common/task/siem_migrations_telemetry_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RuleMigrationTelemetryClient extends _siem_migrations_telemetry_client.SiemMigrationTelemetryClient {
  startSiemMigrationTask() {
    const startTime = Date.now();
    const stats = {
      completed: 0,
      failed: 0
    };
    return {
      startItemTranslation: () => {
        const ruleStartTime = Date.now();
        return {
          success: migrationResult => {
            var _migrationResult$elas;
            stats.completed++;
            this.reportEvent(_siem_migrations.SIEM_MIGRATIONS_RULE_TRANSLATION_SUCCESS, {
              migrationId: this.migrationId,
              translationResult: migrationResult.translation_result || '',
              duration: Date.now() - ruleStartTime,
              model: this.modelName,
              prebuiltMatch: (_migrationResult$elas = migrationResult.elastic_rule) !== null && _migrationResult$elas !== void 0 && _migrationResult$elas.prebuilt_rule_id ? true : false,
              eventName: _siem_migrations.siemMigrationEventNames[_siem_migrations.SiemMigrationsEventTypes.RuleTranslationSuccess]
            });
          },
          failure: error => {
            stats.failed++;
            this.reportEvent(_siem_migrations.SIEM_MIGRATIONS_RULE_TRANSLATION_FAILURE, {
              migrationId: this.migrationId,
              error: error.message,
              model: this.modelName,
              eventName: _siem_migrations.siemMigrationEventNames[_siem_migrations.SiemMigrationsEventTypes.RuleTranslationFailure]
            });
          }
        };
      },
      success: () => {
        const duration = Date.now() - startTime;
        this.reportEvent(_siem_migrations.SIEM_MIGRATIONS_MIGRATION_SUCCESS, {
          migrationId: this.migrationId,
          type: 'rules',
          model: this.modelName || '',
          completed: stats.completed,
          failed: stats.failed,
          total: stats.completed + stats.failed,
          duration,
          eventName: _siem_migrations.siemMigrationEventNames[_siem_migrations.SiemMigrationsEventTypes.MigrationSuccess]
        });
      },
      failure: error => {
        const duration = Date.now() - startTime;
        this.reportEvent(_siem_migrations.SIEM_MIGRATIONS_MIGRATION_FAILURE, {
          migrationId: this.migrationId,
          type: 'rules',
          model: this.modelName || '',
          completed: stats.completed,
          failed: stats.failed,
          total: stats.completed + stats.failed,
          duration,
          error: error.message,
          eventName: _siem_migrations.siemMigrationEventNames[_siem_migrations.SiemMigrationsEventTypes.MigrationFailure]
        });
      },
      aborted: error => {
        const duration = Date.now() - startTime;
        this.reportEvent(_siem_migrations.SIEM_MIGRATIONS_MIGRATION_ABORTED, {
          migrationId: this.migrationId,
          type: 'rules',
          model: this.modelName || '',
          completed: stats.completed,
          failed: stats.failed,
          total: stats.completed + stats.failed,
          duration,
          reason: error.message,
          eventName: _siem_migrations.siemMigrationEventNames[_siem_migrations.SiemMigrationsEventTypes.MigrationAborted]
        });
      }
    };
  }
  reportIntegrationsMatch({
    preFilterIntegrations,
    postFilterIntegration
  }) {
    this.reportEvent(_siem_migrations.SIEM_MIGRATIONS_INTEGRATIONS_MATCH, {
      model: this.modelName,
      migrationId: this.migrationId,
      preFilterIntegrationNames: preFilterIntegrations.map(integration => integration.id) || [],
      preFilterIntegrationCount: preFilterIntegrations.length,
      postFilterIntegrationName: postFilterIntegration ? postFilterIntegration.id : '',
      postFilterIntegrationCount: postFilterIntegration ? 1 : 0,
      eventName: _siem_migrations.siemMigrationEventNames[_siem_migrations.SiemMigrationsEventTypes.RuleTranslationIntegrationsMatch]
    });
  }
  reportPrebuiltRulesMatch({
    preFilterRules,
    postFilterRule
  }) {
    this.reportEvent(_siem_migrations.SIEM_MIGRATIONS_PREBUILT_RULES_MATCH, {
      model: this.modelName,
      migrationId: this.migrationId,
      preFilterRuleNames: preFilterRules.map(rule => rule.rule_id) || [],
      preFilterRuleCount: preFilterRules.length,
      postFilterRuleName: postFilterRule ? postFilterRule.rule_id : '',
      postFilterRuleCount: postFilterRule ? 1 : 0,
      eventName: _siem_migrations.siemMigrationEventNames[_siem_migrations.SiemMigrationsEventTypes.RuleTranslationPrebuiltRulesMatch]
    });
  }
}
exports.RuleMigrationTelemetryClient = RuleMigrationTelemetryClient;