"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchEsQuery = fetchEsQuery;
exports.generateLink = generateLink;
var _common = require("@kbn/triggers-actions-ui-plugin/common");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _esQuery = require("@kbn/es-query");
var _common2 = require("../../../../common");
var _build_sorted_events_query = require("../../../../common/build_sorted_events_query");
var _util = require("../util");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Fetching matching documents for a given rule from elasticsearch by a given index and query
 */
async function fetchEsQuery({
  ruleId,
  name,
  params,
  spacePrefix,
  timestamp,
  services,
  alertLimit,
  dateStart,
  dateEnd,
  sourceFields
}) {
  const {
    scopedClusterClient,
    logger,
    ruleResultService,
    share
  } = services;
  const discoverLocator = share.url.locators.get('DISCOVER_APP_LOCATOR');
  const esClient = scopedClusterClient.asCurrentUser;
  const isGroupAgg = (0, _common.isGroupAggregation)(params.termField);
  const isCountAgg = (0, _common.isCountAggregation)(params.aggType);
  const {
    query,
    fields,
    // eslint-disable-next-line @typescript-eslint/naming-convention
    runtime_mappings,
    _source
  } = (0, _util.getParsedQuery)(params);
  const filter = timestamp && params.excludeHitsFromPreviousRun ? {
    bool: {
      filter: [query, {
        bool: {
          must_not: [{
            bool: {
              filter: [{
                range: {
                  [params.timeField]: {
                    lte: timestamp,
                    format: 'strict_date_optional_time'
                  }
                }
              }]
            }
          }]
        }
      }]
    }
  } : query;
  const sortedQuery = (0, _build_sorted_events_query.buildSortedEventsQuery)({
    index: params.index,
    from: dateStart,
    to: dateEnd,
    filter,
    size: isGroupAgg ? 0 : params.size,
    sortOrder: 'desc',
    searchAfterSortId: undefined,
    timeField: params.timeField,
    track_total_hits: true,
    fields,
    runtime_mappings,
    _source,
    aggs: (0, _common.buildAggregation)({
      aggType: params.aggType,
      aggField: params.aggField,
      termField: params.termField,
      termSize: params.termSize,
      condition: {
        resultLimit: alertLimit,
        conditionScript: (0, _common2.getComparatorScript)(params.thresholdComparator, params.threshold, _common.BUCKET_SELECTOR_FIELD)
      },
      ...(isGroupAgg ? {
        topHitsSize: params.size
      } : {}),
      loggerCb: message => logger.warn(message)
    })
  });
  logger.debug(() => `es query rule ${_ruleDataUtils.ES_QUERY_ID}:${ruleId} "${name}" query - ${JSON.stringify(sortedQuery)}`);
  const {
    body: searchResult
  } = await esClient.search(sortedQuery, {
    meta: true
  });
  logger.debug(() => ` es query rule ${_ruleDataUtils.ES_QUERY_ID}:${ruleId} "${name}" result - ${JSON.stringify(searchResult)}`);

  // result against CCS indices will return success response with errors nested within
  // the _shards or _clusters field; look for these errors and bubble them up
  const anyShardFailures = (0, _util.checkForShardFailures)(searchResult);
  if (anyShardFailures && ruleResultService) {
    ruleResultService.addLastRunWarning(anyShardFailures);
    ruleResultService.setLastRunOutcomeMessage(anyShardFailures);
  }
  const link = generateLink(params, filter, discoverLocator, dateStart, dateEnd, spacePrefix);
  return {
    parsedResults: (0, _common.parseAggregationResults)({
      isCountAgg,
      isGroupAgg,
      esResult: searchResult,
      resultLimit: alertLimit,
      sourceFieldsParams: sourceFields,
      generateSourceFieldsFromHits: true,
      termField: params.termField
    }),
    link,
    index: params.index
  };
}
function generateLink(params, rawFilter, discoverLocator, dateStart, dateEnd, spacePrefix) {
  const dataViewId = 'es_query_rule_adhoc_data_view';
  const filter = (0, _esQuery.buildCustomFilter)(dataViewId, rawFilter, false, false, 'Rule query DSL', _esQuery.FilterStateStore.APP_STATE);
  const redirectUrlParams = {
    dataViewSpec: {
      id: dataViewId,
      title: params.index.join(','),
      timeFieldName: params.timeField
    },
    filters: [filter],
    timeRange: {
      from: dateStart,
      to: dateEnd
    },
    isAlertResults: true
  };

  // use `lzCompress` flag for making the link readable during debugging/testing
  // const redirectUrl = discoverLocator!.getRedirectUrl(redirectUrlParams, { lzCompress: false });
  const redirectUrl = discoverLocator.getRedirectUrl(redirectUrlParams, {
    spaceId: spacePrefix
  });
  return redirectUrl;
}