"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertRenameProcessorToESQL = convertRenameProcessorToESQL;
var _esqlAst = require("@kbn/esql-ast");
var _condition_to_esql = require("../condition_to_esql");
var _common = require("./common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Converts a Streamlang RenameProcessor into a list of ES|QL AST commands.
 *
 * Filters applied for Ingest Pipeline parity:
 * - When `ignore_missing: false`: `WHERE NOT(source_field IS NULL)` filters missing source fields
 * - When `override: false`: `WHERE target_field IS NULL` filters existing target fields
 *
 * Ingest Pipeline throws errors ("field doesn't exist" / "field already exists"),
 * while ES|QL uses filtering to exclude such documents entirely.
 */
function convertRenameProcessorToESQL(processor) {
  const {
    from,
    to,
    // eslint-disable-next-line @typescript-eslint/naming-convention
    ignore_missing = false,
    // default same as Rename Ingest Processor
    override = false,
    // default same as Rename Ingest Processor
    where
  } = processor;
  const fromColumn = _esqlAst.Builder.expression.column(from);
  const toColumn = _esqlAst.Builder.expression.column(to);
  const commands = [];

  // Add missing field filter if needed (ignore_missing = false)
  const missingFieldFilter = (0, _common.buildIgnoreMissingFilter)(from, ignore_missing);
  if (missingFieldFilter) {
    commands.push(missingFieldFilter);
  }

  // Add override filter if needed (override = false)
  const overrideFilter = (0, _common.buildOverrideFilter)(to, override);
  if (overrideFilter) {
    commands.push(overrideFilter);
  }

  // Use the simple RENAME command only for the most basic case (no ignore_missing, no where, and override is true).
  if (!ignore_missing && override && !where) {
    commands.push(_esqlAst.Builder.command({
      name: 'rename',
      args: [_esqlAst.Builder.expression.func.binary('as', [fromColumn, toColumn])]
    }));
    return commands;
  }
  const conditions = [];

  // ignore_missing and override conditions are handled by WHERE to filter out documents
  // this is to be consistent with Ingest Pipeline behavior where such documents are skipped/errored out
  // Processor's `if` condition is handled here as part of the CASE expression below

  if (where) {
    conditions.push((0, _condition_to_esql.conditionToESQLAst)(where));
  }
  const finalCondition = conditions.length > 1 ? conditions.reduce((acc, cond) => _esqlAst.Builder.expression.func.binary('and', [acc, cond])) : conditions[0];
  const assignment = finalCondition ? _esqlAst.Builder.expression.func.call('CASE', [finalCondition, fromColumn, toColumn]) : fromColumn;
  const evalCommand = _esqlAst.Builder.command({
    name: 'eval',
    args: [_esqlAst.Builder.expression.func.binary('=', [toColumn, assignment])]
  });
  const dropCommand = _esqlAst.Builder.command({
    name: 'drop',
    args: [fromColumn]
  });
  commands.push(evalCommand, dropCommand);
  return commands;
}