"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.clusterLogs = clusterLogs;
var _streamlang = require("@kbn/streamlang");
var _util = require("util");
var _pLimit = _interopRequireDefault(require("p-limit"));
var _lodash = require("lodash");
var _cluster_sample_docs = require("./cluster_sample_docs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getFields(condition) {
  if ('field' in condition) {
    return [condition.field];
  }
  if ('and' in condition) {
    return condition.and.flatMap(getFields);
  }
  if ('or' in condition) {
    return condition.or.flatMap(getFields);
  }
  return [];
}

/**
 * Cluster Elasticsearch documents by:
 * - getting 1000 docs per specified partition
 * - per result set, create clusters by using DBSCAN and Jaccard similarity
 *
 * Each cluster has its own document analysis.
 */
async function clusterLogs({
  index,
  partitions,
  esClient,
  start,
  end,
  size = 1000,
  logger,
  dropUnmapped = false
}) {
  var _partitions;
  // time filter
  const rangeQuery = {
    range: {
      '@timestamp': {
        gte: start,
        lte: end,
        format: 'epoch_millis'
      }
    }
  };

  // append an "uncategorized" partition if it does not exist yet
  if (!(0, _lodash.isEqual)((_partitions = partitions[partitions.length - 1]) === null || _partitions === void 0 ? void 0 : _partitions.condition, {
    always: {}
  })) {
    partitions.push({
      name: `Uncategorized logs`,
      condition: {
        always: {}
      }
    });
  }
  const requests = [];

  // extract used fields to create runtime_mappings
  const fieldsToMap = new Set();

  // create requests for exclusive partitions (data only ends up in single bucket)
  partitions.forEach((partition, idx) => {
    const fields = getFields(partition.condition);
    fields.forEach(field => fieldsToMap.add(field));
    const prevPartitions = partitions.slice(0, idx);
    requests.push({
      name: partition.name,
      query: {
        bool: {
          filter: [(0, _streamlang.conditionToQueryDsl)(partition.condition), rangeQuery],
          must_not: prevPartitions.map(prev => (0, _streamlang.conditionToQueryDsl)(prev.condition))
        }
      }
    });
  });

  // get mapped fields for specified index
  const fieldCaps = await esClient.fieldCaps({
    index,
    fields: '*',
    index_filter: {
      bool: {
        filter: [rangeQuery]
      }
    }
  });

  // collect fields that are used in the filters, but unmapped
  const unmappedFields = Array.from(fieldsToMap).filter(field => {
    return fieldCaps.fields[field] === undefined;
  });
  const limiter = (0, _pLimit.default)(5);

  // 1) fetch fieldCaps + random sample
  const sampledDocsResponses = await Promise.all(requests.map(request => {
    return limiter(() => {
      const requestBody = {
        index,
        _source: true,
        size,
        timeout: '5s',
        // add runtime mappings so fields are queryable
        runtime_mappings: dropUnmapped ? {} : Object.fromEntries(Array.from(unmappedFields).map(field => {
          return [field, {
            type: 'keyword'
          }];
        })),
        query: {
          bool: {
            must: [request.query],
            // randomize data set
            should: [{
              function_score: {
                functions: [{
                  random_score: {}
                }]
              }
            }]
          }
        }
      };
      return esClient.search(requestBody);
    });
  }));
  return (0, _lodash.compact)(partitions.map((partition, idx) => {
    const response = sampledDocsResponses[idx];
    if (!response || !response.hits || !response.hits.hits) {
      logger.warn(`Response for ${(0, _util.format)(partition)} expected at ${idx}, but not found (value was ${(0, _util.format)(response)}). Num responses was ${sampledDocsResponses.length}`);
      return null;
    }
    const clustering = (0, _cluster_sample_docs.clusterSampleDocs)({
      hits: response.hits.hits,
      fieldCaps,
      dropUnmapped,
      valueCardinalityLimit: 100
    });
    return {
      name: partition.name,
      condition: partition.condition,
      clustering: {
        ...clustering,
        // drop the samples to reduce the payload
        clusters: clustering.clusters.map(cluster => {
          const {
            samples: _samples,
            ...rest
          } = cluster;
          return rest;
        })
      }
    };
  }));
}