"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleProcessingGrokSuggestions = void 0;
exports.mapFields = mapFields;
exports.processingGrokSuggestionsSchema = void 0;
var _zod = require("@kbn/zod");
var _grokHeuristics = require("@kbn/grok-heuristics");
var _streamsSchema = require("@kbn/streams-schema");
var _otelSemanticConventions = require("@kbn/otel-semantic-conventions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const processingGrokSuggestionsSchema = exports.processingGrokSuggestionsSchema = _zod.z.object({
  path: _zod.z.object({
    name: _zod.z.string()
  }),
  body: _zod.z.object({
    connector_id: _zod.z.string(),
    sample_messages: _zod.z.array(_zod.z.string()),
    review_fields: _zod.z.record(_zod.z.string(), _zod.z.object({
      grok_component: _zod.z.string(),
      example_values: _zod.z.array(_zod.z.string())
    }))
  })
});
const handleProcessingGrokSuggestions = async ({
  params,
  inferenceClient,
  streamsClient,
  fieldsMetadataClient
}) => {
  const stream = await streamsClient.getStream(params.path.name);
  const isWiredStream = _streamsSchema.Streams.WiredStream.Definition.is(stream);
  const response = await inferenceClient.prompt({
    connectorId: params.body.connector_id,
    prompt: _grokHeuristics.ReviewFieldsPrompt,
    input: {
      sample_messages: params.body.sample_messages,
      review_fields: JSON.stringify(params.body.review_fields)
    }
  });
  if (!('toolCalls' in response) || response.toolCalls.length === 0) {
    throw new Error('Failed to generate grok suggestions due to error with the AI generation, please try again later.');
  }
  const reviewResult = response.toolCalls[0].function.arguments;

  // if the stream is wired, or if it matches the logs-*.otel-* pattern, use the OTEL field names
  const useOtelFieldNames = isWiredStream || params.path.name.match(/^logs-.*\.otel-/);
  const fieldMetadata = await fieldsMetadataClient.find({
    fieldNames: reviewResult.fields.map(field => field.ecs_field)
  }).then(fieldsDictionary => fieldsDictionary.toPlain());
  return {
    log_source: reviewResult.log_source,
    fields: mapFields(reviewResult.fields, fieldMetadata, !!useOtelFieldNames)
  };
};
exports.handleProcessingGrokSuggestions = handleProcessingGrokSuggestions;
function mapFields(reviewResults, fieldMetadata, useOtelFieldNames) {
  return reviewResults.map(field => {
    var _fieldMetadata$name$o, _fieldMetadata$name;
    // @timestamp is a special case that we want to map to custom.timestamp - if we let it overwrite @timestamp it will most likely
    // fail because the format won't be right. I a follow-up we can extend the suggestion to also add a date format processor step
    // to map it back correctly.
    const name = field.ecs_field.startsWith('@timestamp') ? field.ecs_field.replace('@timestamp', 'custom.timestamp') : field.ecs_field;
    return {
      // make sure otel field names are translated/prefixed correctly
      name: useOtelFieldNames ? (_fieldMetadata$name$o = (_fieldMetadata$name = fieldMetadata[name]) === null || _fieldMetadata$name === void 0 ? void 0 : _fieldMetadata$name.otel_equivalent) !== null && _fieldMetadata$name$o !== void 0 ? _fieldMetadata$name$o : (0, _otelSemanticConventions.prefixOTelField)(name) : name,
      columns: field.columns,
      grok_components: field.grok_components
    };
  });
}