"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.readStream = readStream;
var _streamsSchema = require("@kbn/streams-schema");
var _esErrors = require("@kbn/es-errors");
var _stream_crud = require("../../../lib/streams/stream_crud");
var _logs_layer = require("../../../lib/streams/component_templates/logs_layer");
var _fields = require("../../../lib/streams/assets/fields");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function readStream({
  name,
  assetClient,
  streamsClient,
  scopedClusterClient
}) {
  const [streamDefinition, {
    [name]: assets
  }] = await Promise.all([streamsClient.getStream(name), assetClient.getAssetLinks([name], ['dashboard', 'rule', 'query'])]);
  const assetsByType = assets.reduce((acc, asset) => {
    const assetType = asset[_fields.ASSET_TYPE];
    if (assetType === 'dashboard') {
      acc.dashboards.push(asset);
    } else if (assetType === 'rule') {
      acc.rules.push(asset);
    } else if (assetType === 'query') {
      acc.queries.push(asset);
    }
    return acc;
  }, {
    dashboards: [],
    rules: [],
    queries: []
  });
  const dashboards = assetsByType.dashboards.map(dashboard => dashboard['asset.id']);
  const rules = assetsByType.rules.map(rule => rule['asset.id']);
  const queries = assetsByType.queries.map(query => {
    return query.query;
  });
  if (_streamsSchema.Streams.GroupStream.Definition.is(streamDefinition)) {
    return {
      stream: streamDefinition,
      dashboards,
      rules,
      queries
    };
  }

  // These queries are only relavant for IngestStreams
  const [ancestors, dataStream, privileges, dataStreamSettings] = await Promise.all([streamsClient.getAncestors(name), streamsClient.getDataStream(name).catch(e => {
    if ((0, _esErrors.isNotFoundError)(e)) {
      return null;
    }
    throw e;
  }), streamsClient.getPrivileges(name), scopedClusterClient.asCurrentUser.indices.getDataStreamSettings({
    name
  }).catch(e => {
    if ((0, _esErrors.isNotFoundError)(e)) {
      return null;
    }
    throw e;
  })]);
  if (_streamsSchema.Streams.ClassicStream.Definition.is(streamDefinition)) {
    return {
      stream: streamDefinition,
      privileges,
      elasticsearch_assets: dataStream && privileges.manage ? await (0, _stream_crud.getUnmanagedElasticsearchAssets)({
        dataStream,
        scopedClusterClient
      }) : undefined,
      data_stream_exists: !!dataStream,
      effective_lifecycle: (0, _stream_crud.getDataStreamLifecycle)(dataStream),
      effective_settings: (0, _stream_crud.getDataStreamSettings)(dataStreamSettings === null || dataStreamSettings === void 0 ? void 0 : dataStreamSettings.data_streams[0]),
      dashboards,
      rules,
      queries
    };
  }
  const inheritedFields = (0, _logs_layer.addAliasesForNamespacedFields)(streamDefinition, (0, _streamsSchema.getInheritedFieldsFromAncestors)(ancestors));
  const body = {
    stream: streamDefinition,
    dashboards,
    rules,
    privileges,
    queries,
    effective_lifecycle: (0, _streamsSchema.findInheritedLifecycle)(streamDefinition, ancestors),
    effective_settings: (0, _streamsSchema.getInheritedSettings)([...ancestors, streamDefinition]),
    inherited_fields: inheritedFields
  };
  return body;
}