"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateMaintenanceWindowRoute = void 0;
var _lib = require("../../../../lib");
var _types = require("../../../../../types");
var _common = require("../../../../../../common");
var _update = require("../../../../../../common/routes/maintenance_window/external/apis/update");
var _response = require("../../../../../../common/routes/maintenance_window/external/response");
var _transforms = require("../common/transforms");
var _transform_update_body = require("./transform_update_body");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateMaintenanceWindowRoute = (router, licenseState) => {
  router.patch({
    path: _types.UPDATE_MAINTENANCE_WINDOW_API_PATH,
    validate: {
      request: {
        body: _update.updateMaintenanceWindowRequestBodySchemaV1,
        params: _update.updateMaintenanceWindowRequestParamsSchemaV1
      },
      response: {
        200: {
          body: () => _response.maintenanceWindowResponseSchemaV1,
          description: 'Indicates a successful call.'
        },
        400: {
          description: 'Indicates an invalid schema or parameters.'
        },
        403: {
          description: 'Indicates that this call is forbidden.'
        },
        404: {
          description: 'Indicates a maintenance window with the given ID does not exist.'
        },
        409: {
          description: 'Indicates that the maintenance window has already been updated by another user.'
        }
      }
    },
    security: {
      authz: {
        requiredPrivileges: [`${_common.MAINTENANCE_WINDOW_API_PRIVILEGES.WRITE_MAINTENANCE_WINDOW}`]
      }
    },
    options: {
      access: 'public',
      summary: 'Update a maintenance window.',
      tags: ['oas-tag:maintenance-window'],
      availability: {
        since: '9.1.0',
        stability: 'stable'
      }
    }
  }, router.handleLegacyErrors((0, _lib.verifyAccessAndContext)(licenseState, async function (context, req, res) {
    licenseState.ensureLicenseForMaintenanceWindow();
    const body = req.body;
    const params = req.params;
    const maintenanceWindowClient = (await context.alerting).getMaintenanceWindowClient();
    const maintenanceWindow = await maintenanceWindowClient.update({
      id: params.id,
      data: (0, _transform_update_body.transformUpdateBodyV1)(body)
    });
    const response = (0, _transforms.transformInternalMaintenanceWindowToExternalV1)(maintenanceWindow);
    return res.ok({
      body: response
    });
  })));
};
exports.updateMaintenanceWindowRoute = updateMaintenanceWindowRoute;