"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ConnectorMappingsService = void 0;
var _constants = require("../../../common/constants");
var _connector_mappings = require("../../common/types/connector_mappings");
var _runtime_types = require("../../common/runtime_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ConnectorMappingsService {
  constructor(log) {
    this.log = log;
  }
  async find({
    unsecuredSavedObjectsClient,
    options
  }) {
    try {
      this.log.debug(`Attempting to find all connector mappings`);
      const connectorMappings = await unsecuredSavedObjectsClient.find({
        ...options,
        type: _constants.CASE_CONNECTOR_MAPPINGS_SAVED_OBJECT
      });
      const validatedMappings = [];
      for (const mapping of connectorMappings.saved_objects) {
        const validatedMapping = (0, _runtime_types.decodeOrThrow)(_connector_mappings.ConnectorMappingsAttributesTransformedRt)(mapping.attributes);
        validatedMappings.push(Object.assign(mapping, {
          attributes: validatedMapping
        }));
      }
      return Object.assign(connectorMappings, {
        saved_objects: validatedMappings
      });
    } catch (error) {
      this.log.error(`Attempting to find all connector mappings: ${error}`);
      throw error;
    }
  }
  async post({
    unsecuredSavedObjectsClient,
    attributes,
    references,
    refresh
  }) {
    try {
      this.log.debug(`Attempting to POST a new connector mappings`);
      const decodedAttributes = (0, _runtime_types.decodeOrThrow)(_connector_mappings.ConnectorMappingsAttributesTransformedRt)(attributes);
      const connectorMappings = await unsecuredSavedObjectsClient.create(_constants.CASE_CONNECTOR_MAPPINGS_SAVED_OBJECT, decodedAttributes, {
        references,
        refresh
      });
      const validatedAttributes = (0, _runtime_types.decodeOrThrow)(_connector_mappings.ConnectorMappingsAttributesTransformedRt)(connectorMappings.attributes);
      return Object.assign(connectorMappings, {
        attributes: validatedAttributes
      });
    } catch (error) {
      this.log.error(`Error on POST a new connector mappings: ${error}`);
      throw error;
    }
  }
  async update({
    unsecuredSavedObjectsClient,
    mappingId,
    attributes,
    references,
    refresh
  }) {
    try {
      this.log.debug(`Attempting to UPDATE connector mappings ${mappingId}`);
      const decodedAttributes = (0, _runtime_types.decodeOrThrow)(_connector_mappings.ConnectorMappingsAttributesPartialRt)(attributes);
      const updatedMappings = await unsecuredSavedObjectsClient.update(_constants.CASE_CONNECTOR_MAPPINGS_SAVED_OBJECT, mappingId, decodedAttributes, {
        references,
        refresh
      });
      const validatedAttributes = (0, _runtime_types.decodeOrThrow)(_connector_mappings.ConnectorMappingsAttributesPartialRt)(updatedMappings.attributes);
      return Object.assign(updatedMappings, {
        attributes: validatedAttributes
      });
    } catch (error) {
      this.log.error(`Error on UPDATE connector mappings ${mappingId}: ${error}`);
      throw error;
    }
  }
}
exports.ConnectorMappingsService = ConnectorMappingsService;