"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PluginWrapper = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _plugin_reader = require("./plugin_reader");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Lightweight wrapper around discovered plugin that is responsible for instantiating
 * plugin and dispatching proper context and dependencies into plugin's lifecycle hooks.
 *
 * @internal
 */
class PluginWrapper {
  constructor(discoveredPlugin, opaqueId, initializerContext) {
    (0, _defineProperty2.default)(this, "name", void 0);
    (0, _defineProperty2.default)(this, "configPath", void 0);
    (0, _defineProperty2.default)(this, "requiredPlugins", void 0);
    (0, _defineProperty2.default)(this, "optionalPlugins", void 0);
    (0, _defineProperty2.default)(this, "runtimePluginDependencies", void 0);
    (0, _defineProperty2.default)(this, "instance", void 0);
    (0, _defineProperty2.default)(this, "startDependencies$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "startDependencies", (0, _rxjs.firstValueFrom)(this.startDependencies$));
    this.discoveredPlugin = discoveredPlugin;
    this.opaqueId = opaqueId;
    this.initializerContext = initializerContext;
    this.name = discoveredPlugin.id;
    this.configPath = discoveredPlugin.configPath;
    this.requiredPlugins = discoveredPlugin.requiredPlugins;
    this.optionalPlugins = discoveredPlugin.optionalPlugins;
    this.runtimePluginDependencies = discoveredPlugin.runtimePluginDependencies;
  }

  /**
   * Instantiates plugin and calls `setup` function exposed by the plugin initializer.
   * @param setupContext Context that consists of various core services tailored specifically
   * for the `setup` lifecycle event.
   * @param plugins The dictionary where the key is the dependency name and the value
   * is the contract returned by the dependency's `setup` function.
   */
  setup(setupContext, plugins) {
    this.instance = this.createPluginInstance();
    return this.instance.setup(setupContext, plugins);
  }

  /**
   * Calls `setup` function exposed by the initialized plugin.
   * @param startContext Context that consists of various core services tailored specifically
   * for the `start` lifecycle event.
   * @param plugins The dictionary where the key is the dependency name and the value
   * is the contract returned by the dependency's `start` function.
   */
  start(startContext, plugins) {
    if (this.instance === undefined) {
      throw new Error(`Plugin "${this.name}" can't be started since it isn't set up.`);
    }
    const startContract = this.instance.start(startContext, plugins);
    this.startDependencies$.next([startContext, plugins, startContract]);
    return startContract;
  }

  /**
   * Calls optional `stop` function exposed by the plugin initializer.
   */
  stop() {
    if (this.instance === undefined) {
      throw new Error(`Plugin "${this.name}" can't be stopped since it isn't set up.`);
    }
    if (typeof this.instance.stop === 'function') {
      this.instance.stop();
    }
    this.instance = undefined;
  }
  createPluginInstance() {
    const initializer = (0, _plugin_reader.read)(this.name);
    const instance = initializer(this.initializerContext);
    if (typeof instance.setup !== 'function') {
      throw new Error(`Instance of plugin "${this.name}" does not define "setup" function.`);
    } else if (typeof instance.start !== 'function') {
      throw new Error(`Instance of plugin "${this.name}" does not define "start" function.`);
    }
    return instance;
  }
}
exports.PluginWrapper = PluginWrapper;