"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SavedObjectsClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _simple_saved_object = require("./simple_saved_object");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const joinUriComponents = (...uriComponents) => uriComponents.filter(comp => Boolean(comp)).map(encodeURIComponent).join('/');

/**
 * Interval that requests are batched for
 * @type {integer}
 */
const BATCH_INTERVAL = 100;
const API_BASE_URL = '/api/saved_objects/';
const getObjectsToFetch = queue => {
  const objects = [];
  const inserted = new Set();
  queue.forEach(({
    id,
    type
  }) => {
    if (!inserted.has(`${type}|${id}`)) {
      objects.push({
        id,
        type
      });
      inserted.add(`${type}|${id}`);
    }
  });
  return objects;
};
const getObjectsToResolve = queue => {
  const responseIndices = [];
  const objectsToResolve = [];
  const inserted = new Map();
  queue.forEach(({
    id,
    type
  }) => {
    const key = `${type}|${id}`;
    const indexForTypeAndId = inserted.get(key);
    if (indexForTypeAndId === undefined) {
      inserted.set(key, objectsToResolve.length);
      responseIndices.push(objectsToResolve.length);
      objectsToResolve.push({
        id,
        type
      });
    } else {
      responseIndices.push(indexForTypeAndId);
    }
  });
  return {
    objectsToResolve,
    responseIndices
  };
};

/**
 * Saved Objects is Kibana's data persistence mechanism allowing plugins to
 * use Elasticsearch for storing plugin state. The client-side
 * SavedObjectsClient is a thin convenience library around the SavedObjects
 * HTTP API for interacting with Saved Objects.
 *
 * @internal
 * @deprecated See https://github.com/elastic/kibana/issues/149098
 */
class SavedObjectsClient {
  /** @internal */
  constructor(http) {
    (0, _defineProperty2.default)(this, "http", void 0);
    (0, _defineProperty2.default)(this, "batchGetQueue", void 0);
    (0, _defineProperty2.default)(this, "batchResolveQueue", void 0);
    /**
     * Throttled processing of get requests into bulk requests at 100ms interval
     */
    (0, _defineProperty2.default)(this, "processBatchGetQueue", (0, _lodash.throttle)(async () => {
      const queue = [...this.batchGetQueue];
      this.batchGetQueue = [];
      try {
        const objectsToFetch = getObjectsToFetch(queue);
        const {
          saved_objects: savedObjects
        } = await this.performBulkGet(objectsToFetch);
        queue.forEach(queueItem => {
          const foundObject = savedObjects.find(savedObject => {
            return savedObject.id === queueItem.id && savedObject.type === queueItem.type;
          });
          if (foundObject) {
            // multiple calls may have been requested the same object.
            // we need to clone to avoid sharing references between the instances
            queueItem.resolve(this.createSavedObject((0, _lodash.cloneDeep)(foundObject)));
          } else {
            queueItem.resolve(this.createSavedObject((0, _lodash.pick)(queueItem, ['id', 'type'])));
          }
        });
      } catch (err) {
        queue.forEach(queueItem => {
          queueItem.reject(err);
        });
      }
    }, BATCH_INTERVAL, {
      leading: false
    }));
    /**
     * Throttled processing of resolve requests into bulk requests at 100ms interval
     */
    (0, _defineProperty2.default)(this, "processBatchResolveQueue", (0, _lodash.throttle)(async () => {
      const queue = [...this.batchResolveQueue];
      this.batchResolveQueue = [];
      try {
        const {
          objectsToResolve,
          responseIndices
        } = getObjectsToResolve(queue);
        const {
          resolved_objects: resolvedObjects
        } = await this.performBulkResolve(objectsToResolve);
        queue.forEach((queueItem, i) => {
          // This differs from the older processBatchGetQueue approach because the resolved object IDs are *not* guaranteed to be the same.
          // Instead, we rely on the guarantee that the objects in the bulkResolve response will be in the same order as the requests.
          // However, we still need to clone the response object because we deduplicate batched requests.
          const responseIndex = responseIndices[i];
          const clone = (0, _lodash.cloneDeep)(resolvedObjects[responseIndex]);
          queueItem.resolve(this.createResolvedSavedObject(clone));
        });
      } catch (err) {
        queue.forEach(queueItem => {
          queueItem.reject(err);
        });
      }
    }, BATCH_INTERVAL, {
      leading: false
    }));
    (0, _defineProperty2.default)(this, "create", (type, attributes, options = {}) => {
      if (!type || !attributes) {
        return Promise.reject(new Error('requires type and attributes'));
      }
      const path = this.getPath([type, options.id]);
      const query = {
        overwrite: options.overwrite
      };
      const createRequest = this.savedObjectsFetch(path, {
        method: 'POST',
        query,
        body: JSON.stringify({
          attributes,
          migrationVersion: options.migrationVersion,
          typeMigrationVersion: options.typeMigrationVersion,
          managed: options.managed,
          references: options.references
        })
      });
      return createRequest.then(resp => this.createSavedObject(resp));
    });
    /**
     * Creates multiple documents at once
     *
     * @param {array} objects - [{ type, id, attributes, references, migrationVersion, typeMigrationVersion, managed }]
     * @param {object} [options={}]
     * @property {boolean} [options.overwrite=false]
     * @returns The result of the create operation containing created saved objects.
     */
    (0, _defineProperty2.default)(this, "bulkCreate", (objects = [], options = {
      overwrite: false
    }) => {
      const path = this.getPath(['_bulk_create']);
      const query = {
        overwrite: options.overwrite
      };
      const request = this.savedObjectsFetch(path, {
        method: 'POST',
        query,
        body: JSON.stringify(objects)
      });
      return request.then(resp => {
        resp.saved_objects = resp.saved_objects.map(d => this.createSavedObject(d));
        return renameKeys({
          saved_objects: 'savedObjects'
        }, resp);
      });
    });
    (0, _defineProperty2.default)(this, "delete", (type, id, options) => {
      if (!type || !id) {
        return Promise.reject(new Error('requires type and id'));
      }
      const query = {
        force: !!(options !== null && options !== void 0 && options.force)
      };
      return this.savedObjectsFetch(this.getPath([type, id]), {
        method: 'DELETE',
        query
      });
    });
    (0, _defineProperty2.default)(this, "bulkDelete", async (objects, options) => {
      const filteredObjects = objects.map(({
        type,
        id
      }) => ({
        type,
        id
      }));
      const queryOptions = {
        force: !!(options !== null && options !== void 0 && options.force)
      };
      const response = await this.performBulkDelete(filteredObjects, queryOptions);
      return {
        statuses: response.statuses
      };
    });
    (0, _defineProperty2.default)(this, "find", options => {
      const path = this.getPath(['_find']);
      const renameMap = {
        defaultSearchOperator: 'default_search_operator',
        fields: 'fields',
        hasReference: 'has_reference',
        hasReferenceOperator: 'has_reference_operator',
        hasNoReference: 'has_no_reference',
        hasNoReferenceOperator: 'has_no_reference_operator',
        page: 'page',
        perPage: 'per_page',
        search: 'search',
        searchFields: 'search_fields',
        sortField: 'sort_field',
        type: 'type',
        filter: 'filter',
        aggs: 'aggs',
        namespaces: 'namespaces',
        preference: 'preference'
      };
      const renamedQuery = renameKeys(renameMap, options);
      const query = _lodash.pick.apply(null, [renamedQuery, ...Object.values(renameMap)]);

      // `has_references` is a structured object. we need to stringify it before sending it, as `fetch`
      // is not doing it implicitly.
      if (query.has_reference) {
        query.has_reference = JSON.stringify(query.has_reference);
      }
      if (query.has_no_reference) {
        query.has_no_reference = JSON.stringify(query.has_no_reference);
      }

      // `aggs` is a structured object. we need to stringify it before sending it, as `fetch`
      // is not doing it implicitly.
      if (query.aggs) {
        query.aggs = JSON.stringify(query.aggs);
      }
      const request = this.savedObjectsFetch(path, {
        method: 'GET',
        query
      });
      return request.then(resp => {
        return renameKeys({
          aggregations: 'aggregations',
          saved_objects: 'savedObjects',
          total: 'total',
          per_page: 'perPage',
          page: 'page'
        }, {
          ...resp,
          saved_objects: resp.saved_objects.map(d => this.createSavedObject(d))
        });
      });
    });
    (0, _defineProperty2.default)(this, "get", (type, id) => {
      if (!type || !id) {
        return Promise.reject(new Error('requires type and id'));
      }
      return new Promise((resolve, reject) => {
        this.batchGetQueue.push({
          type,
          id,
          resolve,
          reject
        });
        this.processBatchGetQueue();
      });
    });
    (0, _defineProperty2.default)(this, "bulkGet", (objects = []) => {
      const filteredObjects = objects.map(obj => (0, _lodash.pick)(obj, ['id', 'type']));
      return this.performBulkGet(filteredObjects).then(resp => {
        resp.saved_objects = resp.saved_objects.map(d => this.createSavedObject(d));
        return renameKeys({
          saved_objects: 'savedObjects'
        }, resp);
      });
    });
    (0, _defineProperty2.default)(this, "resolve", (type, id) => {
      if (!type || !id) {
        return Promise.reject(new Error('requires type and id'));
      }
      return new Promise((resolve, reject) => {
        this.batchResolveQueue.push({
          type,
          id,
          resolve,
          reject
        });
        this.processBatchResolveQueue();
      });
    });
    (0, _defineProperty2.default)(this, "bulkResolve", async (objects = []) => {
      const filteredObjects = objects.map(({
        type,
        id
      }) => ({
        type,
        id
      }));
      const response = await this.performBulkResolve(filteredObjects);
      return {
        resolved_objects: response.resolved_objects.map(resolveResponse => this.createResolvedSavedObject(resolveResponse))
      };
    });
    this.http = http;
    this.batchGetQueue = [];
    this.batchResolveQueue = [];
  }
  async performBulkDelete(objects, queryOptions) {
    const path = this.getPath(['_bulk_delete']);
    const request = this.savedObjectsFetch(path, {
      method: 'POST',
      body: JSON.stringify(objects),
      query: queryOptions
    });
    return request;
  }
  async performBulkGet(objects) {
    const path = this.getPath(['_bulk_get']);
    const request = this.savedObjectsFetch(path, {
      method: 'POST',
      body: JSON.stringify(objects)
    });
    return request;
  }
  async performBulkResolve(objects) {
    const path = this.getPath(['_bulk_resolve']);
    const request = this.savedObjectsFetch(path, {
      method: 'POST',
      body: JSON.stringify(objects)
    });
    return request;
  }
  update(type, id, attributes, {
    version,
    references,
    upsert
  } = {}) {
    if (!type || !id || !attributes) {
      return Promise.reject(new Error('requires type, id and attributes'));
    }
    const path = this.getPath([type, id]);
    const body = {
      attributes,
      references,
      version,
      upsert
    };
    return this.savedObjectsFetch(path, {
      method: 'PUT',
      body: JSON.stringify(body)
    }).then(resp => {
      return this.createSavedObject(resp);
    });
  }
  bulkUpdate(objects = []) {
    const path = this.getPath(['_bulk_update']);
    return this.savedObjectsFetch(path, {
      method: 'PUT',
      body: JSON.stringify(objects)
    }).then(resp => {
      resp.saved_objects = resp.saved_objects.map(d => this.createSavedObject(d));
      return renameKeys({
        saved_objects: 'savedObjects'
      }, resp);
    });
  }
  createSavedObject(options) {
    return new _simple_saved_object.SimpleSavedObjectImpl(this, options);
  }
  createResolvedSavedObject(resolveResponse) {
    const simpleSavedObject = new _simple_saved_object.SimpleSavedObjectImpl(this, resolveResponse.saved_object);
    return {
      saved_object: simpleSavedObject,
      outcome: resolveResponse.outcome,
      alias_target_id: resolveResponse.alias_target_id,
      alias_purpose: resolveResponse.alias_purpose
    };
  }
  getPath(path) {
    return API_BASE_URL + joinUriComponents(...path);
  }

  /**
   * To ensure we don't break backwards compatibility, savedObjectsFetch keeps
   * the old kfetch error format of `{res: {status: number}}` whereas `http.fetch`
   * uses `{response: {status: number}}`.
   */
  savedObjectsFetch(path, {
    method,
    query,
    body
  }) {
    return this.http.fetch(path, {
      method,
      query,
      body
    });
  }
}

/**
 * Returns a new object with the own properties of `obj`, but the
 * keys renamed according to the `keysMap`.
 *
 * @param keysMap - a map of the form `{oldKey: newKey}`
 * @param obj - the object whose own properties will be renamed
 */
exports.SavedObjectsClient = SavedObjectsClient;
const renameKeys = (keysMap, obj) => Object.keys(obj).reduce((acc, key) => {
  return {
    ...acc,
    ...{
      [keysMap[key] || key]: obj[key]
    }
  };
}, {});