"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertDurationToNormalizedEsInterval = convertDurationToNormalizedEsInterval;
exports.convertIntervalToEsInterval = convertIntervalToEsInterval;
exports.getPreciseDurationDescription = getPreciseDurationDescription;
var _moment = _interopRequireDefault(require("moment"));
var _datemath = _interopRequireDefault(require("@kbn/datemath"));
var _i18n = require("@kbn/i18n");
var _utils = require("../../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const unitsDesc = _datemath.default.unitsDesc;
const largeMax = unitsDesc.indexOf('w');
/**
 * Convert a moment.duration into an es
 * compatible expression, and provide
 * associated metadata
 *
 * @param  {moment.duration} duration
 * @return {object}
 */
function convertDurationToNormalizedEsInterval(duration, targetUnit) {
  for (let i = 0; i < unitsDesc.length; i++) {
    const unit = unitsDesc[i];
    const val = duration.as(unit);
    // find a unit that rounds neatly
    if (val >= 1 && Math.floor(val) === val) {
      if (val === 1 && targetUnit && unit !== targetUnit) {
        continue;
      }
      // if the unit is "large", like years, but
      // isn't set to 1 ES will puke. So keep going until
      // we get out of the "large" units
      if (i <= largeMax && val !== 1) {
        continue;
      }
      return {
        value: val,
        unit,
        expression: val + unit
      };
    }
  }
  const ms = duration.as('ms');
  return {
    value: ms,
    unit: 'ms',
    expression: ms + 'ms'
  };
}
function convertIntervalToEsInterval(interval) {
  const {
    value,
    unit
  } = (0, _utils.parseEsInterval)(interval);
  return {
    value,
    unit,
    expression: interval
  };
}
// Below 5 seconds the "humanize" call returns the "few seconds" sentence, which is not ok for ms
// This special config rewrite makes it sure to have precision also for sub-seconds durations
// ref: https://github.com/moment/moment/issues/348
function getPreciseDurationDescription(intervalValue, unit) {
  // moment cannot format anything below seconds, so this requires a manual handling
  if (unit === 'millisecond') {
    return intervalValue === 1 ? _i18n.i18n.translate('data.search.aggs.buckets.intervalOptions.millisecond', {
      defaultMessage: 'millisecond'
    }) : _i18n.i18n.translate('data.search.aggs.buckets.intervalOptions.milliseconds', {
      defaultMessage: '{n} milliseconds',
      values: {
        n: intervalValue
      }
    });
  }
  // Save default values
  const roundingDefault = _moment.default.relativeTimeRounding();
  const units = [{
    unit: 'm',
    value: 60
  },
  // This should prevent to round up 45 minutes to "an hour"
  {
    unit: 's',
    value: 60
  },
  // this should prevent to round up 45 seconds to "a minute"
  {
    unit: 'ss',
    value: 0
  },
  // This should prevent to round anything below 5 seconds to "few seconds"
  {
    unit: 'ms',
    value: 1000
  } // this should render precision at milliseconds level
  ];
  const defaultValues = units.map(({
    unit: u
  }) => _moment.default.relativeTimeThreshold(u));
  const DIGITS = 2;
  const powValue = Math.pow(10, DIGITS);
  _moment.default.relativeTimeRounding(t => {
    return Math.round(t * powValue) / powValue;
  });
  units.forEach(({
    unit: u,
    value
  }) => _moment.default.relativeTimeThreshold(u, value));
  const defaultLocaleConfig = _moment.default.localeData()._config;
  _moment.default.updateLocale(_moment.default.locale(), {
    relativeTime: {
      ss: n => {
        return n === 1 ? _i18n.i18n.translate('data.search.aggs.buckets.intervalOptions.second', {
          defaultMessage: 'second'
        }) : _i18n.i18n.translate('data.search.aggs.buckets.intervalOptions.seconds', {
          defaultMessage: '{n} seconds',
          values: {
            n
          }
        });
      },
      m: _i18n.i18n.translate('data.search.aggs.buckets.intervalOptions.minute', {
        defaultMessage: 'minute'
      }),
      h: _i18n.i18n.translate('data.search.aggs.buckets.intervalOptions.hourly', {
        defaultMessage: 'hour'
      }),
      d: _i18n.i18n.translate('data.search.aggs.buckets.intervalOptions.daily', {
        defaultMessage: 'day'
      }),
      w: _i18n.i18n.translate('data.search.aggs.buckets.intervalOptions.weekly', {
        defaultMessage: 'week'
      }),
      M: _i18n.i18n.translate('data.search.aggs.buckets.intervalOptions.monthly', {
        defaultMessage: 'month'
      }),
      y: _i18n.i18n.translate('data.search.aggs.buckets.intervalOptions.yearly', {
        defaultMessage: 'year'
      })
    }
  });

  // Execute the format/humanize call in the callback
  const result = _moment.default.duration(intervalValue, unit).locale(_i18n.i18n.getLocale()).humanize();

  // restore all the default values now in moment to not break it
  units.forEach(({
    unit: u
  }, i) => _moment.default.relativeTimeThreshold(unit, defaultValues[i]));
  _moment.default.relativeTimeRounding(roundingDefault);

  // restore all the default values now in moment to not break it
  _moment.default.updateLocale(_moment.default.locale(), defaultLocaleConfig);
  return result;
}