"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformToUpdateSchema = exports.transformToCreateSchema = exports.getUpdateScript = exports.createKnowledgeBaseEntry = void 0;
var _uuid = require("uuid");
var _audit_events = require("./audit_events");
var _event_based_telemetry = require("../../lib/telemetry/event_based_telemetry");
var _get_knowledge_base_entry = require("./get_knowledge_base_entry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createKnowledgeBaseEntry = async ({
  auditLogger,
  esClient,
  knowledgeBaseIndex,
  spaceId,
  user,
  knowledgeBaseEntry,
  logger,
  telemetry
}) => {
  var _document$required;
  const createdAt = new Date().toISOString();
  const document = transformToCreateSchema({
    createdAt,
    spaceId,
    user,
    entry: knowledgeBaseEntry
  });
  const telemetryPayload = {
    entryType: document.type,
    required: (_document$required = document.required) !== null && _document$required !== void 0 ? _document$required : false,
    sharing: document.users.length ? 'private' : 'global',
    ...(document.type === 'document' ? {
      source: document.source
    } : {})
  };
  try {
    const response = await esClient.create({
      document,
      id: (0, _uuid.v4)(),
      index: knowledgeBaseIndex,
      refresh: 'wait_for'
    });
    const newKnowledgeBaseEntry = await (0, _get_knowledge_base_entry.getKnowledgeBaseEntry)({
      esClient,
      knowledgeBaseIndex,
      id: response._id,
      logger,
      user
    });
    auditLogger === null || auditLogger === void 0 ? void 0 : auditLogger.log((0, _audit_events.knowledgeBaseAuditEvent)({
      action: _audit_events.KnowledgeBaseAuditAction.CREATE,
      id: newKnowledgeBaseEntry === null || newKnowledgeBaseEntry === void 0 ? void 0 : newKnowledgeBaseEntry.id,
      name: newKnowledgeBaseEntry === null || newKnowledgeBaseEntry === void 0 ? void 0 : newKnowledgeBaseEntry.name,
      outcome: _audit_events.AUDIT_OUTCOME.SUCCESS
    }));
    telemetry.reportEvent(_event_based_telemetry.CREATE_KNOWLEDGE_BASE_ENTRY_SUCCESS_EVENT.eventType, telemetryPayload);
    return newKnowledgeBaseEntry;
  } catch (err) {
    var _err$message;
    logger.error(`Error creating Knowledge Base Entry: ${err} with kbResource: ${knowledgeBaseEntry.name}`);
    auditLogger === null || auditLogger === void 0 ? void 0 : auditLogger.log((0, _audit_events.knowledgeBaseAuditEvent)({
      action: _audit_events.KnowledgeBaseAuditAction.CREATE,
      outcome: _audit_events.AUDIT_OUTCOME.FAILURE,
      error: err
    }));
    telemetry.reportEvent(_event_based_telemetry.CREATE_KNOWLEDGE_BASE_ENTRY_ERROR_EVENT.eventType, {
      ...telemetryPayload,
      errorMessage: (_err$message = err.message) !== null && _err$message !== void 0 ? _err$message : 'Unknown error'
    });
    throw err;
  }
};
exports.createKnowledgeBaseEntry = createKnowledgeBaseEntry;
const transformToUpdateSchema = ({
  user,
  updatedAt,
  entry
}) => {
  var _ref, _user$profile_uid, _entry$required;
  const base = {
    id: entry.id,
    updated_at: updatedAt,
    updated_by: (_ref = (_user$profile_uid = user.profile_uid) !== null && _user$profile_uid !== void 0 ? _user$profile_uid : user.username) !== null && _ref !== void 0 ? _ref : 'unknown',
    name: entry.name,
    type: entry.type,
    global: entry.global,
    users: entry.global ? [] : [{
      id: user.profile_uid,
      name: user.username
    }]
  };
  if (entry.type === 'index') {
    var _restEntry$users, _entry$inputSchema$ma, _entry$inputSchema;
    const {
      inputSchema,
      outputFields,
      queryDescription,
      ...restEntry
    } = entry;
    return {
      ...base,
      ...restEntry,
      users: (_restEntry$users = restEntry.users) !== null && _restEntry$users !== void 0 ? _restEntry$users : base.users,
      query_description: queryDescription,
      input_schema: (_entry$inputSchema$ma = (_entry$inputSchema = entry.inputSchema) === null || _entry$inputSchema === void 0 ? void 0 : _entry$inputSchema.map(schema => ({
        field_name: schema.fieldName,
        field_type: schema.fieldType,
        description: schema.description
      }))) !== null && _entry$inputSchema$ma !== void 0 ? _entry$inputSchema$ma : undefined,
      output_fields: outputFields !== null && outputFields !== void 0 ? outputFields : undefined
    };
  }
  return {
    ...base,
    kb_resource: entry.kbResource,
    required: (_entry$required = entry.required) !== null && _entry$required !== void 0 ? _entry$required : false,
    source: entry.source,
    text: entry.text,
    vector: undefined
  };
};
exports.transformToUpdateSchema = transformToUpdateSchema;
const getUpdateScript = ({
  entry
}) => {
  // Cannot use script for updating documents with semantic_text fields
  return {
    doc: {
      ...entry,
      semantic_text: entry.text
    }
  };
};
exports.getUpdateScript = getUpdateScript;
const transformToCreateSchema = ({
  createdAt,
  spaceId,
  user,
  entry
}) => {
  var _ref2, _user$profile_uid2, _ref3, _user$profile_uid3, _entry$required2;
  const base = {
    '@timestamp': createdAt,
    created_at: createdAt,
    created_by: (_ref2 = (_user$profile_uid2 = user.profile_uid) !== null && _user$profile_uid2 !== void 0 ? _user$profile_uid2 : user.username) !== null && _ref2 !== void 0 ? _ref2 : 'unknown',
    updated_at: createdAt,
    updated_by: (_ref3 = (_user$profile_uid3 = user.profile_uid) !== null && _user$profile_uid3 !== void 0 ? _user$profile_uid3 : user.username) !== null && _ref3 !== void 0 ? _ref3 : 'unknown',
    name: entry.name,
    namespace: spaceId,
    type: entry.type,
    global: entry.global,
    users: entry.global ? [] : [{
      id: user.profile_uid,
      name: user.username
    }]
  };
  if (entry.type === 'index') {
    var _restEntry$users2, _entry$inputSchema$ma2, _entry$inputSchema2;
    const {
      inputSchema,
      outputFields,
      queryDescription,
      ...restEntry
    } = entry;
    return {
      ...base,
      ...restEntry,
      users: (_restEntry$users2 = restEntry.users) !== null && _restEntry$users2 !== void 0 ? _restEntry$users2 : base.users,
      query_description: queryDescription,
      input_schema: (_entry$inputSchema$ma2 = (_entry$inputSchema2 = entry.inputSchema) === null || _entry$inputSchema2 === void 0 ? void 0 : _entry$inputSchema2.map(schema => ({
        field_name: schema.fieldName,
        field_type: schema.fieldType,
        description: schema.description
      }))) !== null && _entry$inputSchema$ma2 !== void 0 ? _entry$inputSchema$ma2 : undefined,
      output_fields: outputFields !== null && outputFields !== void 0 ? outputFields : undefined
    };
  }
  return {
    ...base,
    kb_resource: entry.kbResource,
    required: (_entry$required2 = entry.required) !== null && _entry$required2 !== void 0 ? _entry$required2 : false,
    source: entry.source,
    text: entry.text,
    semantic_text: entry.text
  };
};
exports.transformToCreateSchema = transformToCreateSchema;